function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import { LayerManager, WebMercatorViewport } from 'deck.gl/core';
import spy from './spy';
import gl from './setup-gl';

export function testInitializeLayer(_ref) {
  var layer = _ref.layer,
      viewport = _ref.viewport;

  var layerManager = new LayerManager(gl);
  layerManager.setViewport(new WebMercatorViewport(100, 100));

  try {
    layerManager.setLayers([layer]);
  } catch (error) {
    return error;
  }

  return null;
}

export function testUpdateLayer(_ref2) {
  var layer = _ref2.layer,
      viewport = _ref2.viewport,
      newProps = _ref2.newProps;

  var layerManager = new LayerManager(gl);
  layerManager.setViewport(new WebMercatorViewport(100, 100));

  try {
    layerManager.setLayers([layer]);
    layerManager.setLayers([layer.clone(newProps)]);
  } catch (error) {
    return error;
  }

  return null;
}

export function testDrawLayer(_ref3) {
  var layer = _ref3.layer,
      _ref3$uniforms = _ref3.uniforms,
      uniforms = _ref3$uniforms === undefined ? {} : _ref3$uniforms;

  var layerManager = new LayerManager(gl);
  layerManager.setViewport(new WebMercatorViewport(100, 100));

  try {
    layerManager.setLayers([layer]);
    layerManager.drawLayers();
  } catch (error) {
    return error;
  }

  return null;
}

/**
 * Initialize a layer, test layer update
 * on a series of newProps, assert on the resulting layer
 *
 * Note: Updates are called sequentially. updateProps will be merged
 * with previous props
 *
 * @param {Function} t - test function
 * @param {Object} opt - test options
 * @param {Object} opt.LayerComponent - The layer component class
 * @param {Array} opt.testCases - A list of testCases
 * @param {Object} opt.testCases.INITIAL_PROPS - The initial prop to initialize the layer with
 * @param {Array} opt.testCases.UPDATES - The list of updates to update
 * @param {Object} opt.testCases.UPDATES.updateProps - updated props
 * @param {Function} opt.testCases.UPDATES.assert - callbacks with updated layer, and oldState
 */

export function testLayerUpdates(t, _ref4) {
  var LayerComponent = _ref4.LayerComponent,
      testCases = _ref4.testCases;

  var layerManager = new LayerManager(gl);
  layerManager.setViewport(new WebMercatorViewport(100, 100));

  var newProps = Object.assign({}, testCases.INITIAL_PROPS);
  var layer = new LayerComponent(newProps);

  t.doesNotThrow(function () {
    return layerManager.setLayers([layer]);
  }, 'initialization of ' + LayerComponent.layerName + ' should not fail');

  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    var _loop = function _loop() {
      var _step$value = _step.value,
          updateProps = _step$value.updateProps,
          assert = _step$value.assert;

      // Add on new props every iteration
      Object.assign(newProps, updateProps);

      // copy old state before update
      var oldState = Object.assign({}, layer.state);

      var newLayer = layer.clone(newProps);
      t.doesNotThrow(function () {
        return layerManager.setLayers([newLayer]);
      }, 'update ' + LayerComponent.layerName + ' should not fail');

      // call draw layer
      t.doesNotThrow(function () {
        return layerManager.drawLayers();
      }, 'draw ' + LayerComponent.layerName + ' should not fail');

      // assert on updated layer
      assert(newLayer, oldState, t);
    };

    for (var _iterator = testCases.UPDATES[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      _loop();
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator.return) {
        _iterator.return();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }
}

/**
 * Initialize a parent layer and its subLayer
 * update the parent layer a series of newProps, assert on the updated subLayer
 *
 * Note: Updates are called sequentially. updateProps will be merged
 * with previous props
 *
 * @param {Function} t - test function
 * @param {Object} opt - test options
 * @param {Object} opt.FunctionsToSpy - Functions that spied by spy
 * @param {Object} opt.LayerComponent - The layer component class
 * @param {Array} opt.testCases - A list of testCases
 * @param {Object} opt.testCases.INITIAL_PROPS - The initial prop to initialize the layer with
 * @param {Array} opt.testCases.UPDATES - The list of updates to update
 * @param {Object} opt.testCases.UPDATES.updateProps - updated props
 * @param {Function} opt.testCases.UPDATES.assert - callbacks with updated layer, and oldState
 */

export function testSubLayerUpdateTriggers(t, _ref5) {
  var FunctionsToSpy = _ref5.FunctionsToSpy,
      LayerComponent = _ref5.LayerComponent,
      testCases = _ref5.testCases;

  var layerManager = new LayerManager(gl);
  layerManager.setViewport(new WebMercatorViewport(100, 100));

  var newProps = Object.assign({}, testCases.INITIAL_PROPS);

  // initialize parent layer (generates and initializes)
  var layer = new LayerComponent(newProps);
  t.doesNotThrow(function () {
    return layerManager.setLayers([layer]);
  }, 'initialization of ' + LayerComponent.layerName + ' should not fail');

  // Create a map of spies that the test case can inspect
  var spies = FunctionsToSpy.reduce(function (accu, curr) {
    return Object.assign(accu, _defineProperty({}, curr, spy(LayerComponent.prototype, curr)));
  }, {});

  var _iteratorNormalCompletion2 = true;
  var _didIteratorError2 = false;
  var _iteratorError2 = undefined;

  try {
    var _loop2 = function _loop2() {
      var _step2$value = _step2.value,
          updateProps = _step2$value.updateProps,
          assert = _step2$value.assert;

      // Add on new props every iteration
      Object.assign(newProps, updateProps);

      var newLayer = layer.clone(newProps);
      t.doesNotThrow(function () {
        return layerManager.setLayers([newLayer]);
      }, 'update ' + LayerComponent.layerName + ' should not fail');

      // layer manager should handle match subLayer and tranfer state and props
      // here we assume subLayer matches copy over the new props
      // from a new subLayer
      var subLayer = layer.getSubLayers()[0];

      // assert on updated subLayer
      assert(subLayer, spies, t);

      // reset spies
      Object.keys(spies).forEach(function (k) {
        return spies[k].reset();
      });
    };

    for (var _iterator2 = testCases.UPDATES[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
      _loop2();
    }

    /*
    failures = testInitializeLayer({layer: subLayer});
    t.ok(!failures, `initialize ${LayerComponent.layerName} subLayer should not fail`);
    testCases.UPDATES.reduce((currentProps, {updateProps, assert}) => {
      // merge updated Props with initialProps
      const newProps = Object.assign({}, currentProps, updateProps);
      // call update layer with new props
      testUpdateLayer({layer, newProps});
      testUpdateLayer({layer: subLayer, newProps: newSubLayer.props});
      t.ok(!failures, `update ${LayerComponent.layerName} subLayer should not fail`);
      return newProps;
    }, testCases.INITIAL_PROPS);
    */

    // restore spies
  } catch (err) {
    _didIteratorError2 = true;
    _iteratorError2 = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion2 && _iterator2.return) {
        _iterator2.return();
      }
    } finally {
      if (_didIteratorError2) {
        throw _iteratorError2;
      }
    }
  }

  Object.keys(spies).forEach(function (k) {
    return spies[k].restore();
  });
}

export function testCreateLayer(t, LayerComponent) {
  var props = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

  var failures = false;
  var layer = null;

  try {
    layer = new LayerComponent(Object.assign({
      id: LayerComponent.layerName + '-0'
    }, props));

    t.ok(layer instanceof LayerComponent, LayerComponent.layerName + ' created');
  } catch (error) {
    failures = true;
  }
  t.ok(!failures, 'creating ' + LayerComponent.layerName + ' should not fail');

  return layer;
}

export function testCreateEmptyLayer(t, LayerComponent) {
  var props = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

  var failures = false;
  try {
    var emptyLayer = new LayerComponent(Object.assign({
      id: 'empty' + LayerComponent.layerName,
      data: [],
      pickable: true
    }, props));

    t.ok(emptyLayer instanceof LayerComponent, 'Empty ' + LayerComponent.layerName + ' created');
  } catch (error) {
    failures = true;
  }
  t.ok(!failures, 'creating empty ' + LayerComponent.layerName + ' should not fail');
}

export function testNullLayer(t, LayerComponent) {
  t.doesNotThrow(function () {
    return new LayerComponent({
      id: 'nullPathLayer',
      data: null,
      pickable: true
    });
  }, 'Null ' + LayerComponent.layerName + ' did not throw exception');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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