'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGeojsonFeatures = getGeojsonFeatures;
exports.separateGeojsonFeatures = separateGeojsonFeatures;

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * "Normalizes" complete or partial GeoJSON data into iterable list of features
 * Can accept GeoJSON geometry or "Feature", "FeatureCollection" in addition
 * to plain arrays and iterables.
 * Works by extracting the feature array or wrapping single objects in an array,
 * so that subsequent code can simply iterate over features.
 *
 * @param {object} geojson - geojson data
 * @param {Object|Array} data - geojson object (FeatureCollection, Feature or
 *  Geometry) or array of features
 * @return {Array|"iteratable"} - iterable list of features
 */
function getGeojsonFeatures(geojson) {
  // If array, assume this is a list of features
  if (Array.isArray(geojson)) {
    return geojson;
  }

  (0, _assert2.default)(geojson.type, 'GeoJSON does not have type');

  switch (geojson.type) {
    case 'GeometryCollection':
      (0, _assert2.default)(Array.isArray(geojson.geometries), 'GeoJSON does not have geometries array');
      return geojson.geometries.map(function (geometry) {
        return { geometry: geometry };
      });
    case 'Feature':
      // Wrap the feature in a 'Features' array
      return [geojson];
    case 'FeatureCollection':
      // Just return the 'Features' array from the collection
      (0, _assert2.default)(Array.isArray(geojson.features), 'GeoJSON does not have features array');
      return geojson.features;
    default:
      // Assume it's a geometry, we'll check type in separateGeojsonFeatures
      // Wrap the geometry object in a 'Feature' object and wrap in an array
      return [{ geometry: geojson }];
  }
}

// Linearize
// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

function separateGeojsonFeatures(features) {
  var pointFeatures = [];
  var lineFeatures = [];
  var polygonFeatures = [];
  var polygonOutlineFeatures = [];

  features.forEach(function (feature) {
    (0, _assert2.default)(feature && feature.geometry, 'GeoJSON does not have geometry');

    var _feature$geometry = feature.geometry,
        type = _feature$geometry.type,
        coordinates = _feature$geometry.coordinates,
        properties = feature.properties;

    checkCoordinates(type, coordinates);

    switch (type) {
      case 'Point':
        pointFeatures.push(feature);
        break;
      case 'MultiPoint':
        // TODO - split multipoints
        coordinates.forEach(function (point) {
          pointFeatures.push({ geometry: { coordinates: point }, properties: properties, feature: feature });
        });
        break;
      case 'LineString':
        lineFeatures.push(feature);
        break;
      case 'MultiLineString':
        // Break multilinestrings into multiple lines with same properties
        coordinates.forEach(function (path) {
          lineFeatures.push({ geometry: { coordinates: path }, properties: properties, feature: feature });
        });
        break;
      case 'Polygon':
        polygonFeatures.push(feature);
        // Break polygon into multiple lines with same properties
        coordinates.forEach(function (path) {
          polygonOutlineFeatures.push({ geometry: { coordinates: path }, properties: properties, feature: feature });
        });
        break;
      case 'MultiPolygon':
        // Break multipolygons into multiple polygons with same properties
        coordinates.forEach(function (polygon) {
          polygonFeatures.push({ geometry: { coordinates: polygon }, properties: properties, feature: feature });
          // Break polygon into multiple lines with same properties
          polygon.forEach(function (path) {
            polygonOutlineFeatures.push({ geometry: { coordinates: path }, properties: properties, feature: feature });
          });
        });
        break;
      default:
    }
  });

  return {
    pointFeatures: pointFeatures,
    lineFeatures: lineFeatures,
    polygonFeatures: polygonFeatures,
    polygonOutlineFeatures: polygonOutlineFeatures
  };
}

/**
 * Simple GeoJSON validation util. For perf reasons we do not validate against the full spec,
 * only the following:
   - geometry.type is supported
   - geometry.coordinate has correct nesting level
 */
var COORDINATE_NEST_LEVEL = {
  Point: 1,
  MultiPoint: 2,
  LineString: 2,
  MultiLineString: 3,
  Polygon: 3,
  MultiPolygon: 4
};

function checkCoordinates(type, coordinates) {
  var nestLevel = COORDINATE_NEST_LEVEL[type];

  (0, _assert2.default)(nestLevel, 'Unknown GeoJSON type ' + type);

  while (coordinates && --nestLevel > 0) {
    coordinates = coordinates[0];
  }
  (0, _assert2.default)(coordinates && Number.isFinite(coordinates[0]), type + ' coordinates are malformed');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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