"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

exports.default = "#define SHADER_NAME path-layer-vertex-shader-64\n\nattribute vec3 positions;\n\nattribute vec3 instanceStartPositions;\nattribute vec3 instanceEndPositions;\nattribute vec3 instanceLeftDeltas;\nattribute vec3 instanceRightDeltas;\nattribute float instanceStrokeWidths;\nattribute vec4 instanceColors;\nattribute vec3 instancePickingColors;\nattribute vec2 instanceDashArrays;\n\nuniform float widthScale;\nuniform float widthMinPixels;\nuniform float widthMaxPixels;\nuniform float jointType;\nuniform float miterLimit;\n\nuniform float opacity;\n\nvarying vec4 vColor;\nvarying vec2 vCornerOffset;\nvarying float vMiterLength;\nvarying vec2 vDashArray;\nvarying float vPathPosition;\nvarying float vPathLength;\n\nconst float EPSILON = 0.001;\nconst float PIXEL_EPSILON = 0.1;\n\nfloat flipIfTrue(bool flag) {\n  return -(float(flag) * 2. - 1.);\n}\n\n// calculate line join positions\nvec3 lineJoin(\n  vec3 prevPoint, vec3 currPoint, vec3 nextPoint,\n  float relativePosition, bool isEnd, bool isJoint,\n  float width\n) {\n  vec2 deltaA = currPoint.xy - prevPoint.xy;\n  vec2 deltaB = nextPoint.xy - currPoint.xy;\n\n  float lenA = length(deltaA);\n  float lenB = length(deltaB);\n\n  // when two points are closer than PIXEL_EPSILON in pixels,\n  // assume they are the same point to avoid precision issue\n  lenA = lenA > PIXEL_EPSILON ? lenA : 0.0;\n  lenB = lenB > PIXEL_EPSILON ? lenB : 0.0;\n\n  vec2 dirA = lenA > 0. ? normalize(deltaA) : vec2(0.0, 0.0);\n  vec2 dirB = lenB > 0. ? normalize(deltaB) : vec2(0.0, 0.0);\n\n  vec2 perpA = vec2(-dirA.y, dirA.x);\n  vec2 perpB = vec2(-dirB.y, dirB.x);\n\n  // tangent of the corner\n  vec2 tangent = vec2(dirA + dirB);\n  tangent = length(tangent) > 0. ? normalize(tangent) : perpA;\n  // direction of the corner\n  vec2 miterVec = vec2(-tangent.y, tangent.x);\n  // width offset from current position\n  vec2 perp = isEnd ? perpA : perpB;\n  float L = isEnd ? lenA : lenB;\n\n  // cap super sharp angles\n  float sinHalfA = abs(dot(miterVec, perp));\n  float cosHalfA = abs(dot(dirA, miterVec));\n\n  bool turnsRight = dirA.x * dirB.y > dirA.y * dirB.x;\n\n  float offsetScale = 1.0 / max(sinHalfA, EPSILON);\n\n  float cornerPosition = isJoint ?\n    0.0 :\n    flipIfTrue(turnsRight == (relativePosition > 0.0));\n\n  // do not bevel if line segment is too short\n  cornerPosition *=\n    float(cornerPosition <= 0.0 || sinHalfA < min(lenA, lenB) / width * cosHalfA);\n\n  // trim if inside corner extends further than the line segment\n  if (cornerPosition < 0.0) {\n    offsetScale = min(offsetScale, L / width / max(cosHalfA, EPSILON));\n  }\n\n  vMiterLength = cornerPosition >= 0.0 ?\n    mix(offsetScale, 0.0, cornerPosition) :\n    offsetScale * cornerPosition;\n  vMiterLength -= sinHalfA * jointType;\n\n  float offsetDirection = mix(\n    positions.y,\n    mix(\n      flipIfTrue(turnsRight),\n      positions.y * flipIfTrue(turnsRight == (positions.x == 1.)),\n      cornerPosition\n    ),\n    step(0.0, cornerPosition)\n  );\n\n  vec2 offsetVec = mix(miterVec, -tangent, step(0.5, cornerPosition));\n  offsetScale = mix(offsetScale, 1.0 / max(cosHalfA, 0.001), step(0.5, cornerPosition));\n\n  // special treatment for start cap and end cap\n  // TODO - This has an issue. len is always positive because it is length.\n  // Step returns zero if -lenA<0, so practically this is a comparison of\n  // lenA with zero, with lots of problems because of the -lenA. Can we use EPSILON?\n  bool isStartCap = step(0.0, -lenA) > 0.5;\n  bool isEndCap = step(0.0, -lenB) > 0.5;\n  bool isCap = isStartCap || isEndCap;\n\n  // 0: center, 1: side\n  cornerPosition = isCap ? (1.0 - positions.z) : 0.;\n\n  // start of path: use next - curr\n  if (isStartCap) {\n    offsetVec = mix(dirB, perpB, cornerPosition);\n  }\n\n  // end of path: use curr - prev\n  if (isEndCap) {\n    offsetVec = mix(dirA, perpA, cornerPosition);\n  }\n\n  // extend out a triangle to envelope the round cap\n  if (isCap) {\n    offsetScale = mix(4.0 * jointType, 1.0, cornerPosition);\n    vMiterLength = 1.0 - cornerPosition;\n    offsetDirection = mix(flipIfTrue(isStartCap), positions.y, cornerPosition);\n  }\n\n  vCornerOffset = offsetVec * offsetDirection * offsetScale;\n\n  // Generate variables for dash calculation\n  vDashArray = instanceDashArrays;\n  vPathLength = L / width;\n  // vec2 offsetFromStartOfPath = isEnd ? vCornerOffset + deltaA / width : vCornerOffset;\n  vec2 offsetFromStartOfPath = vCornerOffset;\n  if (isEnd) {\n    offsetFromStartOfPath += deltaA / width;\n  }\n  vec2 dir = isEnd ? dirA : dirB;\n  vPathPosition = dot(offsetFromStartOfPath, dir);\n\n  return currPoint + vec3(vCornerOffset * width, 0.0);\n}\n\n// calculate line join positions\n// extract params from attributes and uniforms\nvec3 lineJoin(vec3 prevPoint, vec3 currPoint, vec3 nextPoint) {\n\n  // relative position to the corner:\n  // -1: inside (smaller side of the angle)\n  // 0: center\n  // 1: outside (bigger side of the angle)\n\n  float relativePosition = positions.y;\n  bool isEnd = positions.x > EPSILON;\n  bool isJoint = positions.z > EPSILON;\n\n  float width = clamp(project_scale(instanceStrokeWidths * widthScale),\n    widthMinPixels, widthMaxPixels) / 2.0;\n\n  return lineJoin(\n    prevPoint, currPoint, nextPoint,\n    relativePosition, isEnd, isJoint,\n    width\n  );\n}\n\nvoid main() {\n  vColor = vec4(instanceColors.rgb, instanceColors.a * opacity) / 255.;\n\n  // Set color to be rendered to picking fbo (also used to check for selection highlight).\n  picking_setPickingColor(instancePickingColors);\n\n  float isEnd = positions.x;\n\n  vec3 prevPosition = mix(-instanceLeftDeltas, vec3(0.0), isEnd) + instanceStartPositions;\n  prevPosition = project_position(prevPosition);\n\n  vec3 currPosition = mix(instanceStartPositions, instanceEndPositions, isEnd);\n  currPosition = project_position(currPosition);\n\n  vec3 nextPosition = mix(vec3(0.0), instanceRightDeltas, isEnd) + instanceEndPositions;\n  nextPosition = project_position(nextPosition);\n\n  vec3 pos = lineJoin(prevPosition, currPosition, nextPosition);\n\n  gl_Position = project_to_clipspace(vec4(pos, 1.0));\n}\n";
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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