'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readFile = readFile;

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) {
  return obj && obj.__esModule ? obj : { default: obj };
}

var window = require('global/window'); // A browser implementation of the Node.js `fs.readFile` method

var File = window.File;

/**
 * File reader function for the browser, intentionally similar
 * to node's fs.readFile API, however returns a Promise rather than
 * callbacks
 *
 * @param {File|Blob} file  HTML File or Blob object to read as string
 * @returns {Promise.string}  Resolves to a string containing file contents
 */
function readFile(file) {
  return new Promise(function (resolve, reject) {
    try {
      (0, _assert2.default)(File, 'window.File not defined. Must run under browser.');
      (0, _assert2.default)(file instanceof File, 'parameter must be a File object');

      var reader = new window.FileReader();

      reader.onerror = function (e) {
        return reject(new Error(getFileErrorMessage(e)));
      };
      reader.onabort = function () {
        return reject(new Error('Read operation was aborted.'));
      };
      reader.onload = function () {
        return resolve(reader.result);
      };

      reader.readAsText(file);
    } catch (error) {
      reject(error);
    }
  });
}

// NOTES ON ERROR HANDLING
//
// Prepared to externalize error message texts
//
// The weird thing about the FileReader API is that the error definitions
// are only available on the error event instance that is passed to the
// handler. Thus we need to create definitions that are avialble outside
// the handler.
//
// https://developer.mozilla.org/en-US/docs/Web/API/FileReader
//
// Side Note: To complicate matters, there are also a DOMError string set on
// filereader object (error property). Not clear how or if these map
// to the event error codes. These strings are not currently used by this api.
//
// https://developer.mozilla.org/en-US/docs/Web/API/DOMError

function getFileErrorMessage(e) {
  // Map event's error codes to static error codes so that we can
  // externalize error code to error message mapping
  switch (e.target.error.code) {
    case e.target.error.NOT_FOUND_ERR:
      return 'File not found';
    case e.target.error.NOT_READABLE_ERR:
      return 'File not readable';
    case e.target.error.ABORT_ERR:
      return 'Aborted';
    case e.target.error.SECURITY_ERR:
      return 'File locked';
    case e.target.error.ENCODING_ERR:
      return 'File too long';
    default:
      return 'Read error';
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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