// Copyright (c) 2015 - 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import {fp64} from 'luma.gl';
import project from '../project/project';
import {fp64ify, fp64ifyMatrix4} from '../../utils/fp64';
import memoize from '../../utils/memoize';

import project64Shader from './project64.glsl';

export default {
  name: 'project64',
  dependencies: [project, fp64],
  vs: project64Shader,
  getUniforms,
  deprecations: [
    {type: 'uniform vec2', old: 'projectionFP64[16]', new: 'project_uViewProjectionMatrixFP64'},
    {type: 'uniform vec2', old: 'projectionScaleFP64', new: 'project64_uScale'}
  ]
};

// TODO - this module should calculate the 64 bit uniforms
// It is currently done by project to minimize duplicated work

const DEFAULT_MODULE_OPTIONS = {};
const getMemoizedUniforms = memoize(calculateUniforms);

function getUniforms(opts = DEFAULT_MODULE_OPTIONS, context = {}) {
  // These two uniforms are generated by the fp32 project module
  const {project_uViewProjectionMatrix, project_uScale} = context;

  if (project_uViewProjectionMatrix && project_uScale) {
    // We only need to update fp64 uniforms if fp32 projection is being updated
    return getMemoizedUniforms({project_uViewProjectionMatrix, project_uScale});
  }
  return {};
}

function calculateUniforms({project_uViewProjectionMatrix, project_uScale}) {
  const glViewProjectionMatrixFP64 = fp64ifyMatrix4(project_uViewProjectionMatrix);
  const scaleFP64 = fp64ify(project_uScale);

  return {
    project_uViewProjectionMatrixFP64: glViewProjectionMatrixFP64,
    project64_uViewProjectionMatrix: glViewProjectionMatrixFP64,
    project64_uScale: scaleFP64
  };
}
