"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.colorDelta = colorDelta;
exports.colorDeltaSq = colorDeltaSq;
exports.diffImagePixels = diffImagePixels;
/*
 * Calculate perceived color difference using YIQ NTSC transmission color space
 * Based on 2010 paper by Yuriy Kotsarenko and Fernando Ramos
 * http://www.progmat.uaem.mx:8080/artVol2Num2/Articulo3Vol2Num2.pdf
 */
var DEFAULT_THRESHOLD = 255 * 0.05;

var getY = function getY(r, g, b) {
  return r * 0.29889531 + g * 0.58662247 + b * 0.11448223;
};
var getI = function getI(r, g, b) {
  return r * 0.59597799 - g * 0.2741761 - b * 0.32180189;
};
var getQ = function getQ(r, g, b) {
  return r * 0.21147017 - g * 0.52261711 + b * 0.31114694;
};
var getESq = function getESq(dY, dI, dQ) {
  return 0.5053 * dY * dY + 0.299 * dI * dI + 0.1957 * dQ * dQ;
};

// Get blended r/g/b value after applying alpha
var applyAlpha = function applyAlpha(c, a) {
  return 255 + (c - 255) * a / 255;
};

/**
 * Get dE square at given index from two pixel arrays
 * @param {Uint8ClampedArray} img1 - pixel data of first image
 * @param {Uint8ClampedArray} img2 - pixel data of second image
 * @param {Number} i - pixel index
 */
function colorDelta(img1, img2, index) {
  return Math.sqrt(colorDeltaSq(img1, img2, index));
}

function colorDeltaSq(img1, img2, index) {
  var i = index * 4;
  var a1 = img1[i + 3];
  var a2 = img2[i + 3];

  var r1 = applyAlpha(img1[i + 0], a1);
  var g1 = applyAlpha(img1[i + 1], a1);
  var b1 = applyAlpha(img1[i + 2], a1);

  var r2 = applyAlpha(img2[i + 0], a2);
  var g2 = applyAlpha(img2[i + 1], a2);
  var b2 = applyAlpha(img2[i + 2], a2);

  return getESq(getY(r1, g1, b1) - getY(r2, g2, b2), getI(r1, g1, b1) - getI(r2, g2, b2), getQ(r1, g1, b1) - getQ(r2, g2, b2));
}

// TODO - expects imagedata structs
// may need a helper func to accept different arguments types
function diffImagePixels(data1, data2) {
  var colorDeltaThreshold = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : DEFAULT_THRESHOLD;

  var pixelCount = data1.data.length / 4;
  var maxDeltaSq = colorDeltaThreshold * colorDeltaThreshold;
  var badPixels = 0;
  for (var i = 0; i < pixelCount; i++) {
    var delta = colorDeltaSq(data1.data, data2.data, i);
    if (delta > maxDeltaSq) {
      badPixels++;
    }
  }
  var percentage = 1 - badPixels / pixelCount;
  return percentage;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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