"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _name = _interopRequireDefault(require("../utils/name"));

var _stringify = _interopRequireDefault(require("../utils/stringify"));

var _reduceAssertionObject = _interopRequireDefault(require("../utils/reduceAssertionObject"));

var _html = _interopRequireDefault(require("../utils/html"));

var _single = _interopRequireDefault(require("../utils/single"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * This source code is licensed under the MIT-style license found in the
 * LICENSE file in the root directory of this source tree. *
 *
 * @providesModule toHaveStyleAssertion
 * 
 */
function flattenStyle(style) {
  if (!style) {
    return undefined;
  }

  if (!Array.isArray(style)) {
    return style;
  }

  return style.reduce((computedStyle, currentStyle) => ({ ...computedStyle,
    ...flattenStyle(currentStyle)
  }), undefined);
}

function toHaveStyle(enzymeWrapper, styleKey, styleValue) {
  const style = flattenStyle(enzymeWrapper.prop('style'));
  const wrapperName = (0, _name.default)(enzymeWrapper); // 1. If the component doesn't have a style prop in general. That's an immediate failure.

  if (!style) {
    return {
      pass: false,
      message: `Expected <${wrapperName}> component to have a style prop but it did not.`,
      negatedMessage: `Expected <${wrapperName}> component not to have a style prop but it did.`,
      contextualInformation: {
        actual: (0, _html.default)(enzymeWrapper)
      }
    };
  } // 2. If the assertion is to check if the component has a style in general. We need to make sure
  // that its not an object and intended for the object assertion API.
  // Then we have to make sure the style has that key.


  if (styleKey === undefined && arguments.length === 2 && typeof styleKey !== 'object' && Array.isArray(styleKey) === false) {
    return {
      pass: style.hasOwnProperty(styleKey),
      message: `Expected <${wrapperName}> to have any value for the prop "${styleKey}"`,
      negatedMessage: `Expected <${wrapperName}> not to receive the prop "${styleKey}"`,
      contextualInformation: {
        actual: `Actual props: ${(0, _stringify.default)({
          [styleKey]: style[styleKey]
        })}`,
        expected: `Expected props: ${(0, _stringify.default)({
          [styleKey]: styleValue
        })}`
      }
    };
  }

  const results = _reduceAssertionObject.default.call(this, style, styleKey, styleValue);

  const unmatchedKeys = results.unmatchedKeys.join(', ');
  const contextualInformation = {
    actual: `Actual style: ${(0, _stringify.default)(results.actual)}`,
    expected: `Expected style: ${(0, _stringify.default)(results.expected)}`
  };

  if (results.missingKeys.length) {
    const missingKeys = results.missingKeys.join(', ');
    return {
      pass: false,
      message: `Expected <${wrapperName}> component to have a style keys of "${missingKeys}" but it did not.`,
      negatedMessage: `Expected <${wrapperName}> component not to have a style key of "${missingKeys}" but it did.`,
      contextualInformation: {
        actual: (0, _html.default)(enzymeWrapper)
      }
    };
  }

  return {
    pass: results.pass,
    message: `Expected <${wrapperName}> component style values to match for key "${unmatchedKeys}", but they didn't`,
    negatedMessage: `Expected <${wrapperName}> component style values to be different for key "${unmatchedKeys}", but they weren't`,
    contextualInformation
  };
}

var _default = (0, _single.default)(toHaveStyle);

exports.default = _default;
module.exports = exports.default;