"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isZeroAddress = exports.zeroAddress = exports.importPublic = exports.privateToPublic = exports.privateToAddress = exports.publicToAddress = exports.pubToAddress = exports.isValidPublic = exports.isValidPrivate = exports.generateAddress2 = exports.generateAddress = exports.isValidChecksumAddress = exports.toChecksumAddress = exports.isValidAddress = exports.Account = void 0;
var assert = require("assert");
var BN = require("bn.js");
var rlp = require("rlp");
var ethjs_util_1 = require("ethjs-util");
var constants_1 = require("./constants");
var bytes_1 = require("./bytes");
var hash_1 = require("./hash");
var helpers_1 = require("./helpers");
var types_1 = require("./types");
var _a = require('ethereum-cryptography/secp256k1'), privateKeyVerify = _a.privateKeyVerify, publicKeyCreate = _a.publicKeyCreate, publicKeyVerify = _a.publicKeyVerify, publicKeyConvert = _a.publicKeyConvert;
var Account = /** @class */ (function () {
    /**
     * This constructor assigns and validates the values.
     * Use the static factory methods to assist in creating an Account from varying data types.
     */
    function Account(nonce, balance, stateRoot, codeHash) {
        if (nonce === void 0) { nonce = new BN(0); }
        if (balance === void 0) { balance = new BN(0); }
        if (stateRoot === void 0) { stateRoot = constants_1.KECCAK256_RLP; }
        if (codeHash === void 0) { codeHash = constants_1.KECCAK256_NULL; }
        this.nonce = nonce;
        this.balance = balance;
        this.stateRoot = stateRoot;
        this.codeHash = codeHash;
        this._validate();
    }
    Account.fromAccountData = function (accountData) {
        var nonce = accountData.nonce, balance = accountData.balance, stateRoot = accountData.stateRoot, codeHash = accountData.codeHash;
        return new Account(nonce ? new BN(bytes_1.toBuffer(nonce)) : undefined, balance ? new BN(bytes_1.toBuffer(balance)) : undefined, stateRoot ? bytes_1.toBuffer(stateRoot) : undefined, codeHash ? bytes_1.toBuffer(codeHash) : undefined);
    };
    Account.fromRlpSerializedAccount = function (serialized) {
        var values = rlp.decode(serialized);
        if (!Array.isArray(values)) {
            throw new Error('Invalid serialized account input. Must be array');
        }
        return this.fromValuesArray(values);
    };
    Account.fromValuesArray = function (values) {
        var nonce = values[0], balance = values[1], stateRoot = values[2], codeHash = values[3];
        return new Account(nonce ? new BN(nonce) : undefined, balance ? new BN(balance) : undefined, stateRoot, codeHash);
    };
    Account.prototype._validate = function () {
        if (this.nonce.lt(new BN(0))) {
            throw new Error('nonce must be greater than zero');
        }
        if (this.balance.lt(new BN(0))) {
            throw new Error('balance must be greater than zero');
        }
        if (this.stateRoot.length !== 32) {
            throw new Error('stateRoot must have a length of 32');
        }
        if (this.codeHash.length !== 32) {
            throw new Error('codeHash must have a length of 32');
        }
    };
    /**
     * Returns a Buffer Array of the raw Buffers for the account, in order.
     */
    Account.prototype.raw = function () {
        return [types_1.bnToRlp(this.nonce), types_1.bnToRlp(this.balance), this.stateRoot, this.codeHash];
    };
    /**
     * Returns the RLP serialization of the account as a `Buffer`.
     */
    Account.prototype.serialize = function () {
        return rlp.encode(this.raw());
    };
    /**
     * Returns a `Boolean` determining if the account is a contract.
     */
    Account.prototype.isContract = function () {
        return !this.codeHash.equals(constants_1.KECCAK256_NULL);
    };
    /**
     * Returns a `Boolean` determining if the account is empty complying to the definition of
     * account emptiness in [EIP-161](https://eips.ethereum.org/EIPS/eip-161):
     * "An account is considered empty when it has no code and zero nonce and zero balance."
     */
    Account.prototype.isEmpty = function () {
        return this.balance.isZero() && this.nonce.isZero() && this.codeHash.equals(constants_1.KECCAK256_NULL);
    };
    return Account;
}());
exports.Account = Account;
/**
 * Checks if the address is a valid. Accepts checksummed addresses too.
 */
exports.isValidAddress = function (hexAddress) {
    helpers_1.assertIsHexString(hexAddress);
    return /^0x[0-9a-fA-F]{40}$/.test(hexAddress);
};
/**
 * Returns a checksummed address.
 *
 * If a eip1191ChainId is provided, the chainId will be included in the checksum calculation. This
 * has the effect of checksummed addresses for one chain having invalid checksums for others.
 * For more details see [EIP-1191](https://eips.ethereum.org/EIPS/eip-1191).
 *
 * WARNING: Checksums with and without the chainId will differ. As of 2019-06-26, the most commonly
 * used variation in Ethereum was without the chainId. This may change in the future.
 */
exports.toChecksumAddress = function (hexAddress, eip1191ChainId) {
    helpers_1.assertIsHexString(hexAddress);
    var address = ethjs_util_1.stripHexPrefix(hexAddress).toLowerCase();
    var prefix = eip1191ChainId !== undefined ? eip1191ChainId.toString() + '0x' : '';
    var hash = hash_1.keccakFromString(prefix + address).toString('hex');
    var ret = '0x';
    for (var i = 0; i < address.length; i++) {
        if (parseInt(hash[i], 16) >= 8) {
            ret += address[i].toUpperCase();
        }
        else {
            ret += address[i];
        }
    }
    return ret;
};
/**
 * Checks if the address is a valid checksummed address.
 *
 * See toChecksumAddress' documentation for details about the eip1191ChainId parameter.
 */
exports.isValidChecksumAddress = function (hexAddress, eip1191ChainId) {
    return exports.isValidAddress(hexAddress) && exports.toChecksumAddress(hexAddress, eip1191ChainId) === hexAddress;
};
/**
 * Generates an address of a newly created contract.
 * @param from The address which is creating this new address
 * @param nonce The nonce of the from account
 */
exports.generateAddress = function (from, nonce) {
    helpers_1.assertIsBuffer(from);
    helpers_1.assertIsBuffer(nonce);
    var nonceBN = new BN(nonce);
    if (nonceBN.isZero()) {
        // in RLP we want to encode null in the case of zero nonce
        // read the RLP documentation for an answer if you dare
        return hash_1.rlphash([from, null]).slice(-20);
    }
    // Only take the lower 160bits of the hash
    return hash_1.rlphash([from, Buffer.from(nonceBN.toArray())]).slice(-20);
};
/**
 * Generates an address for a contract created using CREATE2.
 * @param from The address which is creating this new address
 * @param salt A salt
 * @param initCode The init code of the contract being created
 */
exports.generateAddress2 = function (from, salt, initCode) {
    helpers_1.assertIsBuffer(from);
    helpers_1.assertIsBuffer(salt);
    helpers_1.assertIsBuffer(initCode);
    assert(from.length === 20);
    assert(salt.length === 32);
    var address = hash_1.keccak256(Buffer.concat([Buffer.from('ff', 'hex'), from, salt, hash_1.keccak256(initCode)]));
    return address.slice(-20);
};
/**
 * Checks if the private key satisfies the rules of the curve secp256k1.
 */
exports.isValidPrivate = function (privateKey) {
    return privateKeyVerify(privateKey);
};
/**
 * Checks if the public key satisfies the rules of the curve secp256k1
 * and the requirements of Ethereum.
 * @param publicKey The two points of an uncompressed key, unless sanitize is enabled
 * @param sanitize Accept public keys in other formats
 */
exports.isValidPublic = function (publicKey, sanitize) {
    if (sanitize === void 0) { sanitize = false; }
    helpers_1.assertIsBuffer(publicKey);
    if (publicKey.length === 64) {
        // Convert to SEC1 for secp256k1
        return publicKeyVerify(Buffer.concat([Buffer.from([4]), publicKey]));
    }
    if (!sanitize) {
        return false;
    }
    return publicKeyVerify(publicKey);
};
/**
 * Returns the ethereum address of a given public key.
 * Accepts "Ethereum public keys" and SEC1 encoded keys.
 * @param pubKey The two points of an uncompressed key, unless sanitize is enabled
 * @param sanitize Accept public keys in other formats
 */
exports.pubToAddress = function (pubKey, sanitize) {
    if (sanitize === void 0) { sanitize = false; }
    helpers_1.assertIsBuffer(pubKey);
    if (sanitize && pubKey.length !== 64) {
        pubKey = Buffer.from(publicKeyConvert(pubKey, false).slice(1));
    }
    assert(pubKey.length === 64);
    // Only take the lower 160bits of the hash
    return hash_1.keccak(pubKey).slice(-20);
};
exports.publicToAddress = exports.pubToAddress;
/**
 * Returns the ethereum address of a given private key.
 * @param privateKey A private key must be 256 bits wide
 */
exports.privateToAddress = function (privateKey) {
    return exports.publicToAddress(exports.privateToPublic(privateKey));
};
/**
 * Returns the ethereum public key of a given private key.
 * @param privateKey A private key must be 256 bits wide
 */
exports.privateToPublic = function (privateKey) {
    helpers_1.assertIsBuffer(privateKey);
    // skip the type flag and use the X, Y points
    return Buffer.from(publicKeyCreate(privateKey, false)).slice(1);
};
/**
 * Converts a public key to the Ethereum format.
 */
exports.importPublic = function (publicKey) {
    helpers_1.assertIsBuffer(publicKey);
    if (publicKey.length !== 64) {
        publicKey = Buffer.from(publicKeyConvert(publicKey, false).slice(1));
    }
    return publicKey;
};
/**
 * Returns the zero address.
 */
exports.zeroAddress = function () {
    var addressLength = 20;
    var addr = bytes_1.zeros(addressLength);
    return bytes_1.bufferToHex(addr);
};
/**
 * Checks if a given address is the zero address.
 */
exports.isZeroAddress = function (hexAddress) {
    helpers_1.assertIsHexString(hexAddress);
    var zeroAddr = exports.zeroAddress();
    return zeroAddr === hexAddress;
};
//# sourceMappingURL=account.js.map