// Parameter support.
// Installs definitions that enable querying an object for all its parameters
// with resource.getParameters(). This is mainly useful during debugging.
// Note: Kept separate to avoid bundling in production applications
import GL from '../constants'; // WebGL specification 'types'

export const GLenum = 'GLenum';
export const GLfloat = 'GLfloat';
export const GLint = 'GLint';
export const GLuint = 'GLint';
export const GLboolean = 'GLboolean';
/*
 TODO - will most likely remove some of these fields from the main struct
 but they can be useful for debugging/seer integration, so keep them here for now
export const DBG_PARAMETERS = {
  blend: {
    type: GLboolean,
    params: GL.BLEND,
    value: false,
    setter: (gl, value) => value ? gl.enable(GL.BLEND) : gl.disable(GL.BLEND)
  },

  blendColor: {
    type: new Float32Array(4),
    value: new Float32Array([0, 0, 0, 0]),
    params: GL.BLEND_COLOR,
    setter: (gl, value) => gl.blendColor(...value)
  },

  blendEquation: {
    type: [GLenum, GLenum],
    object: ['rgb', 'alpha'],
    alias: 'blendEquationSeparate',
    value: [GL.FUNC_ADD, GL.FUNC_ADD],
    params: [GL.BLEND_EQUATION_RGB, GL.BLEND_EQUATION_ALPHA],
    setter: (gl, value) => gl.blendEquationSeparate(...value),
    normalizeArgs: args => isArray(args) ? args : [args, args]
  },

  // blend func
  blendFunc: {
    type: [GLenum, GLenum, GLenum, GLenum],
    object: ['srcRgb', 'dstRgb', 'srcAlpha', 'dstAlpha'],
    value: [GL.ONE, GL.ZERO, GL.ONE, GL.ZERO],
    params: [GL.BLEND_SRC_RGB, GL.BLEND_DST_RGB, GL.BLEND_SRC_ALPHA, GL.BLEND_DST_ALPHA],
    setter: (gl, value) => gl.blendFuncSeparate(...value),
    normalizeArgs: args => isArray(args) && args.length === 3 ? [...args, ...args] : args
  },

  clearColor: {
    type: new Float32Array(4),
    params: GL.COLOR_CLEAR_VALUE,
    value: new Float32Array([0, 0, 0, 0]), // TBD
    setter: (gl, value) => gl.clearColor(...value)
  },

  colorMask: {
    type: [GLboolean, GLboolean, GLboolean, GLboolean],
    params: GL.COLOR_WRITEMASK,
    value: [true, true, true, true],
    setter: (gl, value) => gl.colorMask(...value)
  },

  // TODO - We have a name clash here
  cullFace: {
    type: GLboolean,
    params: GL.CULL_FACE,
    value: false,
    setter: (gl, value) => value ? gl.enable(GL.CULL_FACE) : gl.disable(GL.CULL_FACE)
  },

  cullFaceMode: {
    type: GLenum,
    params: GL.CULL_FACE_MODE,
    value: GL.BACK,
    setter: (gl, value) => gl.cullFace(value)
  },

  depthTest: {
    type: GLboolean,
    params: GL.DEPTH_TEST,
    value: false,
    setter: (gl, value) => value ? gl.enable(GL.DEPTH_TEST) : gl.disable(GL.DEPTH_TEST)
  },

  depthClearValue: {
    type: GLfloat,
    params: GL.DEPTH_CLEAR_VALUE,
    value: 1,
    setter: (gl, value) => gl.clearDepth(value)
  },

  depthFunc: {
    type: GLenum,
    params: GL.DEPTH_FUNC,
    value: GL.LESS,
    setter: (gl, value) => gl.depthFunc(value)
  },

  depthRange: {
    type: new Float32Array(2),
    object: ['min', 'max'],
    params: GL.DEPTH_RANGE,
    value: new Float32Array([0, 1]), // TBD
    setter: (gl, value) => gl.depthRange(...value)
  },

  depthWritemask: {
    type: GLboolean,
    params: GL.DEPTH_WRITEMASK,
    value: true,
    setter: (gl, value) => gl.depthMask(value)
  },

  dither: {
    type: GLboolean,
    params: GL.DITHER,
    value: true,
    setter: (gl, value) => value ? gl.enable(GL.DITHER) : gl.disable(GL.DITHER)
  },

  fragmentShaderDerivativeHint: {
    type: GLenum,
    params: GL.FRAGMENT_SHADER_DERIVATIVE_HINT,
    value: GL.DONT_CARE,
    setter: (gl, value) => gl.hint(GL.FRAGMENT_SHADER_DERIVATIVE_HINT, value),
    gl1: 'OES_standard_derivatives'
  },

  frontFace: {
    type: GLenum,
    params: GL.FRONT_FACE,
    value: GL.CCW,
    setter: (gl, value) => gl.frontFace(value)
  },

  // Hint for quality of images generated with glGenerateMipmap
  generateMipmapHint: {
    type: GLenum,
    params: GL.GENERATE_MIPMAP_HINT,
    value: GL.DONT_CARE,
    setter: (gl, value) => gl.hint(GL.GENERATE_MIPMAP_HINT, value)
  },

  lineWidth: {
    type: GLfloat,
    params: GL.LINE_WIDTH,
    value: 1,
    setter: (gl, value) => gl.lineWidth(value)
  },

  polygonOffsetFill: {
    type: GLboolean,
    params: GL.POLYGON_OFFSET_FILL,
    value: false,
    setter: (gl, value) =>
      value ? gl.enable(GL.POLYGON_OFFSET_FILL) : gl.disable(GL.POLYGON_OFFSET_FILL)
  },

  // Add small offset to fragment depth values (by factor × DZ + r × units)
  // Useful for rendering hidden-line images, for applying decals to surfaces,
  // and for rendering solids with highlighted edges.
  // https://www.khronos.org/opengles/sdk/docs/man/xhtml/glPolygonOffset.xml
  polygonOffset: {
    type: [GLfloat, GLfloat],
    object: ['factor', 'units'],
    params: [GL.POLYGON_OFFSET_FACTOR, GL.POLYGON_OFFSET_UNITS],
    value: [0, 0],
    setter: (gl, value) => gl.polygonOffset(...value)
  },

  // TODO - enabling multisampling
  // glIsEnabled with argument GL_SAMPLE_ALPHA_TO_COVERAGE
  // glIsEnabled with argument GL_SAMPLE_COVERAGE

  // specify multisample coverage parameters
  // https://www.khronos.org/opengles/sdk/docs/man/xhtml/glSampleCoverage.xml
  sampleCoverage: {
    type: [GLfloat, GLboolean],
    object: ['value', 'invert'],
    params: [GL.SAMPLE_COVERAGE_VALUE, GL.SAMPLE_COVERAGE_INVERT],
    value: [1.0, false],
    setter: (gl, value) => gl.sampleCoverage(...value)
  },

  scissorTest: {
    type: GLboolean,
    params: GL.SCISSOR_TEST,
    value: false,
    setter: (gl, value) => value ? gl.enable(GL.SCISSOR_TEST) : gl.disable(GL.SCISSOR_TEST)
  },

  scissorBox: {
    type: new Int32Array(4),
    object: ['x', 'y', 'width', 'height'],
    // When scissor test enabled we expect users to set correct scissor box,
    // otherwise we default to following value array.
    params: GL.SCISSOR_BOX,
    value: new Int32Array([0, 0, 1024, 1024]),
    setter: (gl, value) => gl.scissor(...value)
  },

  stencilTest: {
    type: GLboolean,
    params: GL.STENCIL_TEST,
    value: false,
    setter: (gl, value) => value ? gl.enable(GL.STENCIL_TEST) : gl.disable(GL.STENCIL_TEST)
  },

  // Sets index used when stencil buffer is cleared.
  stencilClearValue: {
    type: GLint,
    params: GL.STENCIL_CLEAR_VALUE,
    value: 0,
    setter: (gl, value) => gl.clearStencil(value)
  },

  // Sets bit mask enabling writing of individual bits in the stencil planes
  // https://www.khronos.org/opengles/sdk/docs/man/xhtml/glStencilMaskSeparate.xml
  stencilMask: {
    type: [GLuint, GLuint],
    object: ['mask', 'backMask'],
    value: [0xFFFFFFFF, 0xFFFFFFFF],
    params: [GL.STENCIL_WRITEMASK, GL.STENCIL_BACK_WRITEMASK],
    setter: (gl, value) => {
      value = isArray(value) ? value : [value, value];
      const [mask, backMask] = value;
      gl.stencilMaskSeparate(GL.FRONT, mask);
      gl.stencilMaskSeparate(GL.BACK, backMask);
    }
  },

  // Set stencil testing function, reference value and mask for front and back
  // https://www.khronos.org/opengles/sdk/docs/man/xhtml/glStencilFuncSeparate.xml
  stencilFunc: {
    type: [GLenum, GLint, GLuint, GLenum, GLint, GLuint],
    object: [
      'func', 'ref', 'valueMask', 'backFunc', 'backRef', 'backValueMask'
    ],
    value: [GL.ALWAYS, 0, 0xFFFFFFFF, GL.ALWAYS, 0, 0xFFFFFFFF],
    params: [
      // front
      GL.STENCIL_FUNC,
      GL.STENCIL_REF,
      GL.STENCIL_VALUE_MASK,
      // back
      GL.STENCIL_BACK_FUNC,
      GL.STENCIL_BACK_REF,
      GL.STENCIL_BACK_VALUE_MASK
    ],
    setter: (gl, value) => {
      const [func, ref, mask, backFunc, backRef, backMask] = value;
      gl.stencilFuncSeparate(GL.FRONT, func, ref, mask);
      gl.stencilFuncSeparate(GL.BACK, backFunc, backRef, backMask);
    }
  },

  // Specifies the action to take when the stencil test fails, front and back.
  // Stencil test fail action, depth test fail action, pass action
  // GL.KEEP, GL.ZERO, GL.REPLACE, GL.INCR, GL.INCR_WRAP, GL.DECR, GL.DECR_WRAP,
  // and GL.INVERT
  // https://www.khronos.org/opengles/sdk/docs/man/xhtml/glStencilOpSeparate.xml
  stencilOp: {
    type: [GLenum, GLenum, GLenum, GLenum, GLenum, GLenum],
    object: [
      'fail', 'passDepthFail', 'passDepthPass',
      'backFail', 'backPassDepthFail', 'backPassDepthPass'
    ],
    params: [
      // front
      GL.STENCIL_FAIL,
      GL.STENCIL_PASS_DEPTH_FAIL,
      GL.STENCIL_PASS_DEPTH_PASS,
      // back
      GL.STENCIL_BACK_FAIL,
      GL.STENCIL_BACK_PASS_DEPTH_FAIL,
      GL.STENCIL_BACK_PASS_DEPTH_PASS
    ],
    value: [GL.KEEP, GL.KEEP, GL.KEEP, GL.KEEP, GL.KEEP, GL.KEEP],
    setter: (gl, value) => {
      const [sfail, dpfail, dppass, backSfail, backDpfail, backDppass] = value;
      gl.stencilOpSeparate(GL.FRONT, sfail, dpfail, dppass);
      gl.stencilOpSeparate(GL.BACK, backSfail, backDpfail, backDppass);
    }
  },

  viewport: {
    type: new Int32Array(4),
    object: ['x', 'y', 'width', 'height'],
    // We use [0, 0, 1024, 1024] as default, but usually this is updated in each frame.
    params: GL.VIEWPORT,
    value: new Int32Array([0, 0, 1024, 1024]),
    setter: (gl, value) => gl.viewport(...value)
  },

  // WEBGL1 PIXEL PACK/UNPACK MODES

  // Packing of pixel data in memory (1,2,4,8)
  [GL.PACK_ALIGNMENT]: {
    type: GLint,
    params: GL.PACK_ALIGNMENT,
    value: 4,
    setter: (gl, value) => gl.pixelStorei(GL.PACK_ALIGNMENT, value)
  },
  // Unpacking pixel data from memory(1,2,4,8)
  [GL.UNPACK_ALIGNMENT]: {
    type: GLint,
    params: GL.UNPACK_ALIGNMENT,
    value: 4,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_ALIGNMENT, value)
  },
  // Flip source data along its vertical axis
  [GL.UNPACK_FLIP_Y_WEBGL]: {
    type: GLboolean,
    params: GL.UNPACK_FLIP_Y_WEBGL,
    value: false,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_FLIP_Y_WEBGL, value)
  },
  // Multiplies the alpha channel into the other color channels
  [GL.UNPACK_PREMULTIPLY_ALPHA_WEBGL]: {
    type: GLboolean,
    params: GL.UNPACK_PREMULTIPLY_ALPHA_WEBGL,
    value: false,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_PREMULTIPLY_ALPHA_WEBGL, value)
  },
  // Default color space conversion or no color space conversion.
  [GL.UNPACK_COLORSPACE_CONVERSION_WEBGL]: {
    type: GLenum,
    params: GL.UNPACK_COLORSPACE_CONVERSION_WEBGL,
    value: GL.BROWSER_DEFAULT_WEBGL,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_COLORSPACE_CONVERSION_WEBGL, value)
  },

  // WEBGL2 PIXEL PACK/UNPACK MODES

  // Number of pixels in a row.
  [GL.PACK_ROW_LENGTH]: {
    type: GLint,
    params: GL.PACK_ROW_LENGTH,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.PACK_ROW_LENGTH, value),
    webgl2: true
  },
  // Number of pixels skipped before the first pixel is written into memory.
  [GL.PACK_SKIP_PIXELS]: {
    type: GLint,
    params: GL.PACK_SKIP_PIXELS,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.PACK_SKIP_PIXELS, value),
    webgl2: true
  },
  // Number of rows of pixels skipped before first pixel is written to memory.
  [GL.PACK_SKIP_ROWS]: {
    type: GLint,
    params: GL.PACK_SKIP_ROWS,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.PACK_SKIP_ROWS, value),
    webgl2: true
  },
  // Number of pixels in a row.
  [GL.UNPACK_ROW_LENGTH]: {
    type: GLint,
    params: GL.UNPACK_ROW_LENGTH,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_ROW_LENGTH, value),
    webgl2: true
  },
  // Image height used for reading pixel data from memory
  [GL.UNPACK_IMAGE_HEIGHT]: {
    type: GLint,
    params: GL.UNPACK_IMAGE_HEIGHT,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_IMAGE_HEIGHT, value),
    webgl2: true
  },
  // Number of pixel images skipped before first pixel is read from memory
  [GL.UNPACK_SKIP_PIXELS]: {
    type: GLint,
    params: GL.UNPACK_SKIP_PIXELS,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_SKIP_PIXELS, value),
    webgl2: true
  },
  // Number of rows of pixels skipped before first pixel is read from memory
  [GL.UNPACK_SKIP_ROWS]: {
    type: GLint,
    params: GL.UNPACK_SKIP_ROWS,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_SKIP_ROWS, value),
    webgl2: true
  },
  // Number of pixel images skipped before first pixel is read from memory
  [GL.UNPACK_SKIP_IMAGES]: {
    type: GLint,
    params: GL.UNPACK_SKIP_IMAGES,
    value: 0,
    setter: (gl, value) => gl.pixelStorei(GL.UNPACK_SKIP_IMAGES, value),
    webgl2: true
  }
};
*/

export const BUFFER_PARAMETERS = {
  [GL.BUFFER_SIZE]: {
    webgl1: 0
  },
  // GLint indicating the size of the buffer in bytes.
  [GL.BUFFER_USAGE]: {
    webgl1: 0 // GLenum indicating the usage pattern of the buffer.

  }
};
export const FENCE_SYNC_PARAMETERS = [GL.OBJECT_TYPE, // GLenum, type of sync object (always GL.SYNC_FENCE).
GL.SYNC_STATUS, // GLenum, status of sync object (GL.SIGNALED/GL.UNSIGNALED)
GL.SYNC_CONDITION, // GLenum. object condition (always GL.SYNC_GPU_COMMANDS_COMPLETE).
GL.SYNC_FLAGS // GLenum, flags sync object was created with (always 0)
];
export const FRAMEBUFFER_ATTACHMENT_PARAMETERS = [GL.FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, // WebGLRenderbuffer or WebGLTexture
GL.FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE, // GL.RENDERBUFFER, GL.TEXTURE, GL.NONE
GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE, // GL.TEXTURE_CUBE_MAP_POSITIVE_X, etc.
GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL, // GLint
// EXT_sRGB or WebGL2
GL.FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING, // GL.LINEAR, GL.SRBG
// WebGL2
GL.FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER, // GLint
GL.FRAMEBUFFER_ATTACHMENT_RED_SIZE, // GLint
GL.FRAMEBUFFER_ATTACHMENT_GREEN_SIZE, // GLint
GL.FRAMEBUFFER_ATTACHMENT_BLUE_SIZE, // GLint
GL.FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE, // GLint
GL.FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE, // GLint
GL.FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE, // GLint
GL.FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE // GL.FLOAT, GL.INT, GL.UNSIGNED_INT, GL.SIGNED_NORMALIZED, OR GL.UNSIGNED_NORMALIZED.
];
export const FRAMEBUFFER_STATUS = {
  [GL.FRAMEBUFFER_COMPLETE]: 'Success. Framebuffer is correctly set up',
  [GL.FRAMEBUFFER_INCOMPLETE_ATTACHMENT]: 'Framebuffer attachment types mismatched or some attachment point not attachment complete',
  [GL.FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT]: 'Framebuffer has no attachment',
  [GL.FRAMEBUFFER_INCOMPLETE_DIMENSIONS]: 'Framebuffer attachments do not have the same size',
  [GL.FRAMEBUFFER_UNSUPPORTED]: 'Framebuffer attachment format not supported or depth and stencil attachments are not same',
  // When using a WebGL 2 context, the following values can be returned
  [GL.FRAMEBUFFER_INCOMPLETE_MULTISAMPLE]: 'Framebuffer attachement SAMPLES differs among renderbuffers, or are mixed with textures'
};
export const PROGRAM_PARAMETERS = {
  [GL.DELETE_STATUS]: {
    webgl1: 0
  },
  // GLboolean
  [GL.LINK_STATUS]: {
    webgl1: 0
  },
  // GLboolean
  [GL.VALIDATE_STATUS]: {
    webgl1: 0
  },
  // GLboolean
  [GL.ATTACHED_SHADERS]: {
    webgl1: 0
  },
  // GLint
  [GL.ACTIVE_ATTRIBUTES]: {
    webgl1: 0
  },
  // GLint
  [GL.ACTIVE_UNIFORMS]: {
    webgl1: 0
  },
  // GLint
  [GL.TRANSFORM_FEEDBACK_BUFFER_MODE]: {
    webgl2: 0
  },
  // SEPARATE_ATTRIBS/INTERLEAVED_ATTRIBS
  [GL.TRANSFORM_FEEDBACK_VARYINGS]: {
    webgl2: 0
  },
  // GLint
  [GL.ACTIVE_UNIFORM_BLOCKS]: {
    webgl2: 0 // GLint

  }
}; //  parameters

export const RENDERBUFFER_PARAMETERS = {
  // WebGL1 parameters
  [GL.RENDERBUFFER_WIDTH]: {
    webgl1: 0
  },
  // {GLint} - height of the image of renderbuffer.
  [GL.RENDERBUFFER_HEIGHT]: {
    webgl1: 0
  },
  // {GLint} - height of the image of renderbuffer.
  // Internal format of the currently bound renderbuffer.
  // The default is GL.RGBA4. Possible return values:
  // GL.RGBA4: 4 red bits, 4 green bits, 4 blue bits 4 alpha bits.
  // GL.RGB565: 5 red bits, 6 green bits, 5 blue bits.
  // GL.RGB5_A1: 5 red bits, 5 green bits, 5 blue bits, 1 alpha bit.
  // GL.DEPTH_COMPONENT16: 16 depth bits.
  // GL.STENCIL_INDEX8: 8 stencil bits.
  [GL.RENDERBUFFER_INTERNAL_FORMAT]: {
    type: 'GLenum',
    webgl1: GL.RGBA4
  },
  [GL.RENDERBUFFER_GREEN_SIZE]: {
    webgl1: 0
  },
  // {GLint} - resolution (bits) of green color
  [GL.RENDERBUFFER_BLUE_SIZE]: {
    webgl1: 0
  },
  // {GLint} - resolution (bits) of blue color
  [GL.RENDERBUFFER_RED_SIZE]: {
    webgl1: 0
  },
  // {GLint} - resolution (bits) of red color
  [GL.RENDERBUFFER_ALPHA_SIZE]: {
    webgl1: 0
  },
  // {GLint} - resolution (bits) of alpha component
  [GL.RENDERBUFFER_DEPTH_SIZE]: {
    webgl1: 0
  },
  // {GLint} - resolution (bits) of depth component
  [GL.RENDERBUFFER_STENCIL_SIZE]: {
    webgl1: 0
  },
  // {GLint} - resolution (bits) of stencil component
  // When using a WebGL 2 context, the following value is available
  [GL.RENDERBUFFER_SAMPLES]: {
    webgl2: 1
  }
};
export const SAMPLER_PARAMETERS = {
  [GL.TEXTURE_MAG_FILTER]: {
    webgl2: true
  },
  // texture magnification filter
  [GL.TEXTURE_MIN_FILTER]: {
    webgl2: true
  },
  // texture minification filter
  [GL.TEXTURE_WRAP_S]: {
    webgl2: true
  },
  // texture wrapping function for texture coordinate s
  [GL.TEXTURE_WRAP_T]: {
    webgl2: true
  },
  // texture wrapping function for texture coordinate t
  [GL.TEXTURE_WRAP_R]: {
    webgl2: true
  },
  // texture wrapping function for texture coordinate r
  [GL.TEXTURE_BASE_LEVEL]: {
    webgl2: true
  },
  // Texture mipmap level
  [GL.TEXTURE_MAX_LEVEL]: {
    webgl2: true
  },
  // Maximum texture mipmap array level
  [GL.TEXTURE_COMPARE_FUNC]: {
    webgl2: true
  },
  // texture comparison function
  [GL.TEXTURE_COMPARE_MODE]: {
    webgl2: true
  },
  // texture comparison mode
  [GL.TEXTURE_MIN_LOD]: {
    webgl2: true
  },
  // minimum level-of-detail value
  [GL.TEXTURE_MAX_LOD]: {
    webgl2: true // maximum level-of-detail value
    // [GL.TEXTURE_MAX_ANISOTROPY_EXT]: {webgl2: 'EXT_texture_filter_anisotropic'}

  }
}; // const SHADER_PARAMETERS = [
//   GL.DELETE_STATUS, // GLboolean - whether shader is flagged for deletion.
//   GL.COMPILE_STATUS, // GLboolean - was last shader compilation successful.
//   GL.SHADER_TYPE // GLenum - GL.VERTEX_SHADER or GL.FRAGMENT_SHADER.
// ];

export const TEXTURE_PARAMETERS = {
  // WEBGL1
  [GL.TEXTURE_MAG_FILTER]: {
    type: 'GLenum',
    webgl1: GL.LINEAR
  },
  // texture magnification filter
  [GL.TEXTURE_MIN_FILTER]: {
    type: 'GLenum',
    webgl1: GL.NEAREST_MIPMAP_LINEAR
  },
  // minification filt.
  [GL.TEXTURE_WRAP_S]: {
    type: 'GLenum',
    webgl1: GL.REPEAT
  },
  // texture wrapping for coordinate s
  [GL.TEXTURE_WRAP_T]: {
    type: 'GLenum',
    webgl1: GL.REPEAT
  },
  // texture wrapping for texture t
  // WEBGL2
  [GL.TEXTURE_WRAP_R]: {
    type: 'GLenum',
    webgl2: GL.REPEAT
  },
  // texture wrapping for texture r
  [GL.TEXTURE_BASE_LEVEL]: {
    webgl2: 0
  },
  // Texture mipmap level
  [GL.TEXTURE_MAX_LEVEL]: {
    webgl2: 1000
  },
  // Maximum texture mipmap array level
  [GL.TEXTURE_COMPARE_FUNC]: {
    type: 'GLenum',
    webgl2: GL.LEQUAL
  },
  // texture comparison function
  [GL.TEXTURE_COMPARE_MODE]: {
    type: 'GLenum',
    webgl2: GL.NONE
  },
  // texture comparison mode
  [GL.TEXTURE_MIN_LOD]: {
    webgl2: -1000
  },
  // minimum level-of-detail value
  [GL.TEXTURE_MAX_LOD]: {
    webgl2: 1000
  },
  // maximum level-of-detail value
  // WebGL Extensions
  // [GL.TEXTURE_MAX_ANISOTROPY_EXT]: {webgl1: 1.0, extension: 'EXT_texture_filter_anisotropic'},
  // Emulated parameters - These OpenGL parameters are not supported by OpenGL ES
  [GL.TEXTURE_WIDTH]: {
    webgl1: 0
  },
  [GL.TEXTURE_HEIGHT]: {
    webgl1: 0
  }
}; // TODO - separate install parameter definitions function from api metadata

/*
import Buffer from '../buffer';
import FenceSync from '../fence-sync';
import Framebuffer from '../framebuffer';
import Program from '../program';
import Renderbuffer from '../renderbuffer';
import Sampler from '../sampler';
// import Shader from './shader';
import Texture from '../texture';
*/

export function installParameterDefinitions() {// Buffer.PARAMETERS = BUFFER_PARAMETERS;
  // FenceSync.PARAMETERS = FENCE_SYNC_PARAMETERS;
  // Framebuffer.ATTACHMENT_PARAMETERS = FRAMEBUFFER_ATTACHMENT_PARAMETERS;
  // Framebuffer.STATUS = FRAMEBUFFER_STATUS;
  // Program.PARAMETERS = PROGRAM_PARAMETERS;
  // Renderbuffer.PARAMETERS = RENDERBUFFER_PARAMETERS;
  // Sampler.PARAMETERS = SAMPLER_PARAMETERS;
  // // Shader.PARAMETERS = SHADER_PARAMETERS;
  // Texture.PARAMETERS = TEXTURE_PARAMETERS;
}
//# sourceMappingURL=debug-parameters.js.map