import GL from '../constants';
import Resource from './resource';
import { assertWebGL2Context } from '../webgl-utils';
import queryManager from '../webgl-utils/query-manager';
export default class FenceSync extends Resource {
  /*
   * @class
   * @param {WebGL2RenderingContext} gl
   */
  constructor(gl, opts) {
    assertWebGL2Context(gl);
    super(gl, opts); // query manager needs a promise field

    this.promise = null;
    Object.seal(this);
  }
  /**
   * The method is a no-op in the absence of the possibility of
   * synchronizing between multiple GL contexts.
   * Prevent commands from being added to GPU command queue.
   * Note: commands can still be buffered in driver.
   *
   * @param {GLbitfield} flags
   * @param {GLint64} timeout
   * @return {Sync} status
   */


  wait({
    flags = 0,
    timeout = GL.TIMEOUT_IGNORED
  } = {}) {
    this.gl.waitSync(this.handle, flags, timeout);
    return this;
  }
  /**
   * Block all CPU operations until fence is signalled
   * @param {GLbitfield} flags
   * @param {GLint64} timeout
   * @return {GLenum} result
   */


  clientWait({
    flags = GL.SYNC_FLUSH_COMMANDS_BIT,
    timeout
  }) {
    const result = this.gl.clientWaitSync(this.handle, flags, timeout); // TODO - map to boolean?

    switch (result) {
      case GL.ALREADY_SIGNALED:
        // Indicates that sync object was signaled when this method was called.
        break;

      case GL.TIMEOUT_EXPIRED:
        // Indicates that timeout time passed, sync object did not become signaled
        break;

      case GL.CONDITION_SATISFIED:
        // Indicates that sync object was signaled before timeout expired.
        break;

      case GL.WAIT_FAILED:
        // Indicates that an error occurred during execution.
        break;

      default:
    }

    return result;
  }

  cancel() {
    queryManager.cancelQuery(this);
  }

  isSignaled() {
    return this.getParameter(GL.SYNC_STATUS) === GL.SIGNALED;
  } // TODO - Query manager needs these?


  isResultAvailable() {
    return this.isSignaled();
  }

  getResult() {
    return this.isSignaled();
  }

  getParameter(pname) {
    return this.gl.getSyncParameter(this.handle, pname);
  } // PRIVATE METHODS


  _createHandle() {
    return this.gl.fenceSync(GL.SYNC_GPU_COMMANDS_COMPLETE, 0);
  }

  _deleteHandle() {
    queryManager.deleteQuery(this);
    this.gl.deleteSync(this.handle);
  }

}
//# sourceMappingURL=fence-sync.js.map