"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fp64ify = fp64ify;
exports.fp64LowPart = fp64LowPart;
exports.fp64ifyMatrix4 = fp64ifyMatrix4;

/**
 * Calculate WebGL 64 bit float
 * @param a {number} - the input float number
 * @param out {array, optional} - the output array. If not supplied, a new array is created.
 * @param startIndex {integer, optional} - the index in the output array to fill from. Default 0.
 * @returns {array} - the fp64 representation of the input number
 */
function fp64ify(a) {
  var out = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
  var startIndex = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
  var hiPart = Math.fround(a);
  var loPart = a - hiPart;
  out[startIndex] = hiPart;
  out[startIndex + 1] = loPart;
  return out;
}
/**
 * Calculate the low part of a WebGL 64 bit float
 * @param a {number} - the input float number
 * @returns {number} - the lower 32 bit of the number
 */


function fp64LowPart(a) {
  return a - Math.fround(a);
}
/**
 * Calculate WebGL 64 bit matrix (transposed "Float64Array")
 * @param matrix {Matrix4} - the input matrix
 * @returns {array} - the fp64 representation of the input matrix
 */


function fp64ifyMatrix4(matrix) {
  // Transpose the projection matrix to column major for GLSL.
  var matrixFP64 = new Float32Array(32);

  for (var i = 0; i < 4; ++i) {
    for (var j = 0; j < 4; ++j) {
      var index = i * 4 + j;
      fp64ify(matrix[j * 4 + i], matrixFP64, index * 2);
    }
  }

  return matrixFP64;
}
//# sourceMappingURL=fp64-utils.js.map