"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isWebGL = isWebGL;
exports.isWebGL2 = isWebGL2;
exports.assertWebGLContext = assertWebGLContext;
exports.assertWebGL2Context = assertWebGL2Context;
exports.setContextDefaults = setContextDefaults;
exports.createGLContext = createGLContext;
exports.destroyGLContext = destroyGLContext;
exports.resizeGLContext = resizeGLContext;
exports.pollGLContext = pollGLContext;
exports.ERR_WEBGL2 = exports.ERR_WEBGL = exports.ERR_CONTEXT = void 0;

var _webglUtils = require("../webgl-utils");

var _createHeadlessContext = require("./create-headless-context");

var _createCanvas = require("./create-canvas");

var _createBrowserContext = require("./create-browser-context");

var _trackContextState = _interopRequireDefault(require("./track-context-state"));

var _debugContext = require("./debug-context");

var _contextLimits = require("./context-limits");

var _queryManager = _interopRequireDefault(require("../webgl-utils/query-manager"));

var _utils = require("../utils");

var _assert = _interopRequireDefault(require("../utils/assert"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _instanceof(left, right) { if (right != null && typeof Symbol !== "undefined" && right[Symbol.hasInstance]) { return right[Symbol.hasInstance](left); } else { return left instanceof right; } }

// Heuristic testing of contexts (to indentify debug wrappers around gl contexts)
var GL_ARRAY_BUFFER = 0x8892;
var GL_TEXTURE_BINDING_3D = 0x806A;
var ERR_CONTEXT = 'Invalid WebGLRenderingContext';
exports.ERR_CONTEXT = ERR_CONTEXT;
var ERR_WEBGL = ERR_CONTEXT;
exports.ERR_WEBGL = ERR_WEBGL;
var ERR_WEBGL2 = 'Requires WebGL2';
exports.ERR_WEBGL2 = ERR_WEBGL2;

function isWebGL(gl) {
  return Boolean(gl && (_instanceof(gl, _webglUtils.WebGLRenderingContext) || gl.ARRAY_BUFFER === GL_ARRAY_BUFFER));
}

function isWebGL2(gl) {
  return Boolean(gl && (_instanceof(gl, _webglUtils.WebGL2RenderingContext) || gl.TEXTURE_BINDING_3D === GL_TEXTURE_BINDING_3D));
}

function assertWebGLContext(gl) {
  // Need to handle debug context
  (0, _assert.default)(isWebGL(gl), ERR_CONTEXT);
}

function assertWebGL2Context(gl) {
  // Need to handle debug context
  (0, _assert.default)(isWebGL2(gl), ERR_WEBGL2);
}

var contextDefaults = {
  // COMMON CONTEXT PARAMETERS
  // Attempt to allocate WebGL2 context
  webgl2: true,
  // Attempt to create a WebGL2 context (false to force webgl1)
  webgl1: true,
  // Attempt to create a WebGL1 context (false to fail if webgl2 not available)
  throwOnFailure: true,
  manageState: true,
  // BROWSER CONTEXT PARAMETERS
  canvas: null,
  // A canvas element or a canvas string id
  debug: false,
  // Instrument context (at the expense of performance)
  // HEADLESS CONTEXT PARAMETERS
  width: 800,
  // width are height are only used by headless gl
  height: 600 // WEBGL/HEADLESS CONTEXT PARAMETERS
  // Remaining options are passed through to context creator

};
/*
 * Change default context creation parameters.
 * Main use case is regression test suite.
 */

function setContextDefaults() {
  var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  Object.assign(contextDefaults, {
    width: 1,
    height: 1
  }, opts);
}
/*
 * Creates a context giving access to the WebGL API
 */

/* eslint-disable complexity, max-statements */


function createGLContext() {
  var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  opts = Object.assign({}, contextDefaults, opts);
  var _opts = opts,
      canvas = _opts.canvas,
      width = _opts.width,
      height = _opts.height,
      throwOnError = _opts.throwOnError,
      manageState = _opts.manageState,
      debug = _opts.debug; // Error reporting function, enables exceptions to be disabled

  function onError(message) {
    if (throwOnError) {
      throw new Error(message);
    } // log.log(0, message)();


    return null;
  }

  var gl;

  if (_utils.isBrowser) {
    // Get or create a canvas
    var targetCanvas = (0, _createCanvas.getCanvas)({
      canvas: canvas,
      width: width,
      height: height,
      onError: onError
    }); // Create a WebGL context in the canvas

    gl = (0, _createBrowserContext.createBrowserContext)({
      canvas: targetCanvas,
      opts: opts
    });
  } else {
    // Create a headless-gl context under Node.js
    gl = (0, _createHeadlessContext.createHeadlessContext)({
      width: width,
      height: height,
      opts: opts,
      onError: onError
    });
  }

  if (!gl) {
    return null;
  } // Install context state tracking


  if (manageState) {
    (0, _trackContextState.default)(gl, {
      copyState: false,
      log: function log() {
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
          args[_key] = arguments[_key];
        }

        return _utils.log.log.apply(_utils.log, [1].concat(args))();
      }
    });
  } // Add debug instrumentation to the context


  if (_utils.isBrowser && debug) {
    gl = (0, _debugContext.makeDebugContext)(gl, {
      debug: debug
    }); // Debug forces log level to at least 1

    _utils.log.priority = Math.max(_utils.log.priority, 1); // Log some debug info about the context
  } // Log context information


  logInfo(gl); // Add to seer integration

  return gl;
}

function destroyGLContext(gl) {
  // TODO - Remove from seer integration
  // TODO - Unregister any tracking/polyfills
  // There is no way to delete browser based context
  // Destroy headless gl context
  var ext = gl.getExtension('STACKGL_destroy_context');

  if (ext) {
    ext.destroy();
  }
}
/**
 * Resize the canvas' drawing buffer.
 *
 * Can match the canvas CSS size, and optionally also consider devicePixelRatio
 * Can be called every frame
 *
 * Regardless of size, the drawing buffer will always be scaled to the viewport, but
 * for best visual results, usually set to either:
 *  canvas CSS width x canvas CSS height
 *  canvas CSS width * devicePixelRatio x canvas CSS height * devicePixelRatio
 * See http://webgl2fundamentals.org/webgl/lessons/webgl-resizing-the-canvas.html
 *
 * resizeGLContext(gl, {width, height, useDevicePixels})
 */


function resizeGLContext(gl) {
  var opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  // Resize browser context
  if (gl.canvas) {
    /* global window */
    var devicePixelRatio = opts.useDevicePixels ? window.devicePixelRatio || 1 : 1;
    var width = "width" in opts ? opts.width : gl.canvas.clientWidth;
    var height = "height" in opts ? opts.height : gl.canvas.clientHeight;
    gl.canvas.width = width * devicePixelRatio;
    gl.canvas.height = height * devicePixelRatio;
    return;
  } // Resize headless gl context


  var ext = gl.getExtension('STACKGL_resize_drawingbuffer');

  if (ext && "width" in opts && "height" in opts) {
    ext.resize(opts.width, opts.height);
  }
} // POLLING FOR PENDING QUERIES
// Calling this function checks all pending queries for completion


function pollGLContext(gl) {
  _queryManager.default.poll(gl);
} // HELPER METHODS


function logInfo(gl) {
  var webGL = isWebGL2(gl) ? 'WebGL2' : 'WebGL1';
  var info = (0, _contextLimits.glGetDebugInfo)(gl);
  var driver = info ? "(".concat(info.vendor, ",").concat(info.renderer, ")") : '';
  var debug = gl.debug ? ' debug' : '';

  _utils.log.once(0, "".concat(webGL).concat(debug, " context ").concat(driver))();
}
//# sourceMappingURL=context.js.map