/* eslint-disable no-inline-comments, max-len */
import GL from '../constants';
import Resource from './resource';
import Buffer from './buffer';
import { withParameters } from '../webgl-context/context-state';
import { WebGLBuffer } from '../webgl-utils';
import { isWebGL2, assertWebGL2Context, getKey } from '../webgl-utils';
import { log, uid, isPowerOfTwo } from '../utils';
import assert from '../utils/assert'; // Supported min filters for NPOT texture.

const NPOT_MIN_FILTERS = [GL.LINEAR, GL.NEAREST]; // const S3TC = 'WEBGL_compressed_texture_s3tc';
// const PVRTC = 'WEBGL_compressed_texture_pvrtc';
// const ES3 = 'WEBGL_compressed_texture_es3';
// const ETC1 = 'WEBGL_compressed_texture_etc1';
// const SRGB = 'EXT_sRGB';
// const DEPTH = 'WEBGL_depth_texture';
// Legal combinations for internalFormat, format and type

export const TEXTURE_FORMATS = {
  // Unsized texture format - more performance
  [GL.RGB]: {
    dataFormat: GL.RGB,
    types: [GL.UNSIGNED_BYTE, GL.UNSIGNED_SHORT_5_6_5]
  },
  [GL.RGBA]: {
    dataFormat: GL.RGBA,
    types: [GL.UNSIGNED_BYTE, GL.UNSIGNED_SHORT_4_4_4_4, GL.UNSIGNED_SHORT_5_5_5_1]
  },
  [GL.ALPHA]: {
    dataFormat: GL.ALPHA,
    types: [GL.UNSIGNED_BYTE]
  },
  [GL.LUMINANCE]: {
    dataFormat: GL.LUMINANCE,
    types: [GL.UNSIGNED_BYTE]
  },
  [GL.LUMINANCE_ALPHA]: {
    dataFormat: GL.LUMINANCE_ALPHA,
    types: [GL.UNSIGNED_BYTE] // [GL.DEPTH_COMPONENT]: {types: [GL.UNSIGNED_SHORT, GL.UNSIGNED_INT, GL.UNSIGNED_INT_24_8], gl1: DEPTH},
    // [GL.DEPTH_STENCIL]: {gl1: DEPTH},
    // Sized texture format - more performance
    // R
    // [GL.R8]: {dataFormat: GL.RED, types: [GL.UNSIGNED_BYTE], gl2: true},
    // [GL.R16F]: {dataFormat: GL.RED, types: [GL.HALF_FLOAT, GL.FLOAT], gl2: true},
    // [GL.R32F]: {dataFormat: GL.RED, types: [GL.FLOAT], gl2: true},
    // [GL.R8UI]: {dataFormat: GL.RED_INTEGER, types: [GL.UNSIGNED_BYTE], gl2: true},
    // // RG
    // [GL.RG8]: {dataFormat: GL.RG, types: [GL.UNSIGNED_BYTE], gl2: true},
    // [GL.RG16F]: {dataFormat: GL.RG, types: [GL.HALF_FLOAT, GL.FLOAT], gl2: true},
    // [GL.RG32F]: {dataFormat: GL.RG, types: [GL.FLOAT], gl2: true},
    // [GL.RG8UI]: {dataFormat: GL.RG_INTEGER, types: [GL.UNSIGNED_BYTE], gl2: true},
    // // RGB
    // [GL.RGB8]: {dataFormat: GL.RGB, types: [GL.UNSIGNED_BYTE], gl2: true, gl1: SRGB},
    // [GL.SRGB8]: {dataFormat: GL.RGB, types: [GL.UNSIGNED_BYTE], gl2: true, gl1: SRGB},
    // [GL.RGB565]: {dataFormat: GL.RGB, types: [GL.UNSIGNED_BYTE, GL.UNSIGNED_SHORT_5_6_5], gl2: true},
    // [GL.R11F_G11F_B10F]: {dataFormat: GL.RGB, types: [GL.UNSIGNED_INT_10F_11F_11F_REV, GL.HALF_FLOAT, GL.FLOAT], gl2: true},
    // [GL.RGB9_E5]: {dataFormat: GL.RGB, types: [GL.HALF_FLOAT, GL.FLOAT], gl2: true, gl1: 'WEBGL_color_buffer_half_float'},
    // [GL.RGB16F]: {dataFormat: GL.RGB, types: [GL.HALF_FLOAT, GL.FLOAT], gl2: true, gl1: 'WEBGL_color_buffer_float'},
    // [GL.RGB32F]: {dataFormat: GL.RGB, types: [GL.FLOAT], gl2: true},
    // [GL.RGB8UI]: {dataFormat: GL.RGB_INTEGER, types: [GL.UNSIGNED_BYTE], gl2: true},
    // // RGBA
    // [GL.RGBA8]: {dataFormat: GL.RGBA, types: [GL.UNSIGNED_BYTE], gl2: true, gl1: SRGB},
    // [GL.SRGB8_ALPHA8]: {dataFormat: GL.RGBA, types: [GL.UNSIGNED_BYTE], gl2: true, gl1: SRGB},
    // [GL.RGB5_A1]: {dataFormat: GL.RGBA, types: [GL.UNSIGNED_BYTE, GL.UNSIGNED_SHORT_5_5_5_1], gl2: true},
    // [GL.RGBA4]: {dataFormat: GL.RGBA, types: [GL.UNSIGNED_BYTE, GL.UNSIGNED_SHORT_4_4_4_4], gl2: true},
    // [GL.RGBA16F]: {dataFormat: GL.RGBA, types: [GL.HALF_FLOAT, GL.FLOAT], gl2: true},
    // [GL.RGBA32F]: {dataFormat: GL.RGBA, types: [GL.FLOAT], gl2: true},
    // [GL.RGBA8UI]: {dataFormat: GL.RGBA_INTEGER, types: [GL.UNSIGNED_BYTE], gl2: true}
    // Compressed formats
    // WEBGL_compressed_texture_s3tc
    // [GL.COMPRESSED_RGB_S3TC_DXT1_EXT]: {compressed: true, gl1: S3TC},
    // [GL.COMPRESSED_RGBA_S3TC_DXT1_EXT]: {compressed: true, gl1: S3TC},
    // [GL.COMPRESSED_RGBA_S3TC_DXT3_EXT]: {compressed: true, gl1: S3TC},
    // [GL.COMPRESSED_RGBA_S3TC_DXT5_EXT]: {compressed: true, gl1: S3TC},
    // WEBGL_compressed_texture_es3
    // [GL.COMPRESSED_R11_EAC]: {compressed: true, gl1: ES3}, // RED
    // [GL.COMPRESSED_SIGNED_R11_EAC]: {compressed: true, gl1: ES3}, // RED
    // [GL.COMPRESSED_RG11_EAC]: {compressed: true, gl1: ES3}, // RG
    // [GL.COMPRESSED_SIGNED_RG11_EAC]: {compressed: true, gl1: ES3}, // RG
    // [GL.COMPRESSED_RGB8_ETC2]: {compressed: true, gl1: ES3}, // RGB
    // [GL.COMPRESSED_RGBA8_ETC2_EAC]: {compressed: true, gl1: ES3}, // RBG
    // [GL.COMPRESSED_SRGB8_ETC2]: {compressed: true, gl1: ES3}, // RGB
    // [GL.COMPRESSED_SRGB8_ALPHA8_ETC2_EAC]: {compressed: true, gl1: ES3}, // RGBA
    // [GL.COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2]: {compressed: true, gl1: ES3}, // RGBA
    // [GL.COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2]: {compressed: true, gl1: ES3}, // RGBA

    /* WebGL2 guaranteed availability compressed formats?
    COMPRESSED_R11_EAC RED
    COMPRESSED_SIGNED_R11_EAC RED
    COMPRESSED_RG11_EAC RG
    COMPRESSED_SIGNED_RG11_EAC RG
    COMPRESSED_RGB8_ETC2 RGB
    COMPRESSED_SRGB8_ETC2 RGB
    COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2 RGBA
    COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2 RGBA
    COMPRESSED_RGBA8_ETC2_EAC RGBA
    COMPRESSED_SRGB8_ALPHA8_ETC2_EAC
    */
    // WEBGL_compressed_texture_pvrtc
    // [GL.COMPRESSED_RGB_PVRTC_4BPPV1_IMG]: {compressed: true, gl1: PVRTC},
    // [GL.COMPRESSED_RGBA_PVRTC_4BPPV1_IMG]: {compressed: true, gl1: PVRTC},
    // [GL.COMPRESSED_RGB_PVRTC_2BPPV1_IMG]: {compressed: true, gl1: PVRTC},
    // [GL.COMPRESSED_RGBA_PVRTC_2BPPV1_IMG]: {compressed: true, gl1: PVRTC},
    // WEBGL_compressed_texture_etc1
    // [GL.COMPRESSED_RGB_ETC1_WEBGL]: {compressed: true, gl1: ETC1},
    // WEBGL_compressed_texture_atc
    // [GL.COMPRESSED_RGB_ATC_WEBGL]: {compressed: true, gl1: ETC1},
    // [GL.COMPRESSED_RGBA_ATC_EXPLICIT_ALPHA_WEBGL]: {compressed: true, gl1: ETC1},
    // [GL.COMPRESSED_RGBA_ATC_INTERPOLATED_ALPHA_WEBGL]: {compressed: true, gl1: ETC1}

  }
};

function isFormatSupported(gl, format) {
  const info = TEXTURE_FORMATS[format];

  if (!info) {
    return false;
  }

  if (info.gl1 === undefined && info.gl2 === undefined) {
    // No info - always supported
    return true;
  }

  const value = isWebGL2(gl) ? info.gl2 || info.gl1 : info.gl1;
  return typeof value === 'string' ? gl.getExtension(value) : value;
}

function isLinearFilteringSupported(gl, format) {
  const info = TEXTURE_FORMATS[format];

  switch (info && info.types[0]) {
    // Both WebGL1 and WebGL2?
    case GL.FLOAT:
      return gl.getExtension('OES_texture_float_linear');
    // Not in WebGL2?

    case GL.HALF_FLOAT:
      return gl.getExtension('OES_texture_half_float_linear');

    default:
      return true;
  }
}

export default class Texture extends Resource {
  static isSupported(gl, {
    format,
    linearFiltering
  } = {}) {
    let supported = true;

    if (format) {
      supported = supported && isFormatSupported(gl, format);
      supported = supported && (!linearFiltering || isLinearFilteringSupported(gl, format));
    }

    return supported;
  } // target cannot be modified by bind:
  // textures are special because when you first bind them to a target,
  // they get special information. When you first bind a texture as a
  // GL_TEXTURE_2D, you are actually setting special state in the texture.
  // You are saying that this texture is a 2D texture.
  // And it will always be a 2D texture; this state cannot be changed ever.
  // If you have a texture that was first bound as a GL_TEXTURE_2D,
  // you must always bind it as a GL_TEXTURE_2D;
  // attempting to bind it as GL_TEXTURE_1D will give rise to an error
  // (while run-time).


  constructor(gl, opts) {
    const _opts$id = opts.id,
          id = _opts$id === void 0 ? uid('texture') : _opts$id,
          handle = opts.handle,
          target = opts.target;
    super(gl, {
      id,
      handle
    });
    this.target = target;
    this.hasFloatTexture = gl.getExtension('OES_texture_float');
    this.textureUnit = undefined;
  }

  toString() {
    return `Texture(${this.id},${this.width}x${this.height})`;
  }
  /* eslint-disable max-len, max-statements */


  initialize(opts = {}) {
    let data = opts.data;
    const _opts$pixels = opts.pixels,
          pixels = _opts$pixels === void 0 ? null : _opts$pixels,
          _opts$format = opts.format,
          format = _opts$format === void 0 ? GL.RGBA : _opts$format,
          _opts$type = opts.type,
          type = _opts$type === void 0 ? GL.UNSIGNED_BYTE : _opts$type,
          _opts$border = opts.border,
          border = _opts$border === void 0 ? 0 : _opts$border,
          _opts$recreate = opts.recreate,
          recreate = _opts$recreate === void 0 ? false : _opts$recreate,
          _opts$parameters = opts.parameters,
          parameters = _opts$parameters === void 0 ? {} : _opts$parameters,
          _opts$pixelStore = opts.pixelStore,
          pixelStore = _opts$pixelStore === void 0 ? {} : _opts$pixelStore,
          _opts$unpackFlipY = opts.unpackFlipY,
          unpackFlipY = _opts$unpackFlipY === void 0 ? true : _opts$unpackFlipY;
    let _opts$mipmaps = opts.mipmaps,
        mipmaps = _opts$mipmaps === void 0 ? true : _opts$mipmaps; // pixels variable is for API compatibility purpose

    if (!data) {
      // TODO - This looks backwards? Commenting out for now until we decide
      // which prop to use
      // log.deprecated('data', 'pixels')();
      data = pixels;
    }

    let width = opts.width,
        height = opts.height,
        dataFormat = opts.dataFormat; // Deduce width and height

    var _deduceParameters = this._deduceParameters({
      format,
      type,
      dataFormat,
      compressed: false,
      data,
      width,
      height
    });

    width = _deduceParameters.width;
    height = _deduceParameters.height;
    dataFormat = _deduceParameters.dataFormat;
    // Store opts for accessors
    this.width = width;
    this.height = height;
    this.format = format;
    this.type = type;
    this.dataFormat = dataFormat;
    this.border = border;
    this.mipmaps = mipmaps; // Note: luma.gl defaults to GL.UNPACK_FLIP_Y_WEBGL = true;
    // TODO - compare v4 and v3

    const DEFAULT_TEXTURE_SETTINGS = {
      // Pixel store
      [GL.UNPACK_FLIP_Y_WEBGL]: unpackFlipY
    };
    const glSettings = Object.assign({}, DEFAULT_TEXTURE_SETTINGS, pixelStore);

    if (this._isNPOT()) {
      log.warn(`texture: ${this} is Non-Power-Of-Two, disabling mipmaping`)();
      mipmaps = false;

      this._updateForNPOT(parameters);
    }

    this.setImageData({
      data,
      width,
      height,
      format,
      type,
      dataFormat,
      border,
      mipmaps,
      parameters: glSettings
    });

    if (mipmaps) {
      this.generateMipmap();
    } // Set texture sampler parameters


    this.setParameters(parameters); // TODO - Store data to enable auto recreate on context loss

    if (recreate) {
      this.data = data;
    }
  } // If size has changed, reinitializes with current format
  // note clears image and mipmaps


  resize({
    width,
    height
  }) {
    if (width !== this.width || height !== this.height) {
      return this.initialize({
        width,
        height,
        format: this.format,
        type: this.type,
        dataFormat: this.dataFormat,
        border: this.border,
        mipmaps: false
      });
    }

    return this;
  } // Call to regenerate mipmaps after modifying texture(s)


  generateMipmap(params = {}) {
    this.gl.bindTexture(this.target, this.handle);
    withParameters(this.gl, params, () => {
      this.gl.generateMipmap(this.target);
    });
    this.gl.bindTexture(this.target, null);
    return this;
  }
  /*
   * Allocates storage
   * @param {*} pixels -
   *  null - create empty texture of specified format
   *  Typed array - init from image data in typed array
   *  Buffer|WebGLBuffer - (WEBGL2) init from image data in WebGLBuffer
   *  HTMLImageElement|Image - Inits with content of image. Auto width/height
   *  HTMLCanvasElement - Inits with contents of canvas. Auto width/height
   *  HTMLVideoElement - Creates video texture. Auto width/height
   *
   * @param {GLint} width -
   * @param {GLint} height -
   * @param {GLint} mipMapLevel -
   * @param {GLenum} format - format of image data.
   * @param {GLenum} type
   *  - format of array (autodetect from type) or
   *  - (WEBGL2) format of buffer
   * @param {Number} offset - (WEBGL2) offset from start of buffer
   * @param {GLint} border - must be 0.
   * @parameters - temporary settings to be applied, can be used to supply pixel store settings.
   */

  /* eslint-disable max-len, max-statements, complexity */


  setImageData({
    target = this.target,
    pixels = null,
    data = null,
    width,
    height,
    level = 0,
    format = GL.RGBA,
    type,
    dataFormat,
    offset = 0,
    border = 0,
    compressed = false,
    parameters = {}
  }) {
    // pixels variable is  for API compatibility purpose
    if (!data) {
      data = pixels;
    }

    var _deduceParameters2 = this._deduceParameters({
      format,
      type,
      dataFormat,
      compressed,
      data,
      width,
      height
    });

    type = _deduceParameters2.type;
    dataFormat = _deduceParameters2.dataFormat;
    compressed = _deduceParameters2.compressed;
    width = _deduceParameters2.width;
    height = _deduceParameters2.height;
    const gl = this.gl;
    gl.bindTexture(this.target, this.handle);
    let dataType = null;

    var _getDataType = this._getDataType({
      data,
      compressed
    });

    data = _getDataType.data;
    dataType = _getDataType.dataType;
    withParameters(this.gl, parameters, () => {
      switch (dataType) {
        case 'null':
          gl.texImage2D(target, level, format, width, height, border, dataFormat, type, data);
          break;

        case 'typed-array':
          // Looks like this assert is not necessary, as offset is ignored under WebGL1
          // assert((offset === 0 || isWebGL2(gl)), 'offset supported in WebGL2 only');
          gl.texImage2D(target, level, format, width, height, border, dataFormat, type, data, offset);
          break;

        case 'buffer':
          // WebGL2 enables creating textures directly from a WebGL buffer
          assertWebGL2Context(gl);
          gl.bindBuffer(GL.PIXEL_UNPACK_BUFFER, data.handle || data);
          gl.texImage2D(target, level, format, width, height, border, format, type, offset);
          break;

        case 'browser-object':
          gl.texImage2D(target, level, format, format, type, data);
          break;

        case 'compressed':
          gl.compressedTexImage2D(this.target, level, format, width, height, border, data);
          break;

        default:
          assert(false, 'Unknown image data type');
      }
    });
  }
  /* eslint-enable max-len, max-statements, complexity */

  /**
   * Redefines an area of an existing texture
   * Note: does not allocate storage
   */

  /*
   * Redefines an area of an existing texture
   * @param {*} pixels, data -
   *  null - create empty texture of specified format
   *  Typed array - init from image data in typed array
   *  Buffer|WebGLBuffer - (WEBGL2) init from image data in WebGLBuffer
   *  HTMLImageElement|Image - Inits with content of image. Auto width/height
   *  HTMLCanvasElement - Inits with contents of canvas. Auto width/height
   *  HTMLVideoElement - Creates video texture. Auto width/height
   *
   * @param {GLint} x - xOffset from where texture to be updated
   * @param {GLint} y - yOffset from where texture to be updated
   * @param {GLint} width - width of the sub image to be updated
   * @param {GLint} height - height of the sub image to be updated
   * @param {GLint} level - mip level to be updated
   * @param {GLenum} format - internal format of image data.
   * @param {GLenum} type
   *  - format of array (autodetect from type) or
   *  - (WEBGL2) format of buffer or ArrayBufferView
   * @param {GLenum} dataFormat - format of image data.
   * @param {Number} offset - (WEBGL2) offset from start of buffer
   * @param {GLint} border - must be 0.
   * @parameters - temporary settings to be applied, can be used to supply pixel store settings.
   */


  setSubImageData({
    target = this.target,
    pixels = null,
    data = null,
    x = 0,
    y = 0,
    width,
    height,
    level = 0,
    format = GL.RGBA,
    type,
    dataFormat,
    compressed = false,
    offset = 0,
    border = 0,
    parameters = {}
  }) {
    var _deduceParameters3 = this._deduceParameters({
      format,
      type,
      dataFormat,
      compressed,
      data,
      width,
      height
    });

    type = _deduceParameters3.type;
    dataFormat = _deduceParameters3.dataFormat;
    compressed = _deduceParameters3.compressed;
    width = _deduceParameters3.width;
    height = _deduceParameters3.height;

    // pixels variable is  for API compatibility purpose
    if (!data) {
      data = pixels;
    } // Support ndarrays


    if (data && data.data) {
      const ndarray = data;
      data = ndarray.data;
      width = ndarray.shape[0];
      height = ndarray.shape[1];
    } // Support buffers


    if (data instanceof Buffer) {
      data = data.handle;
    }

    this.gl.bindTexture(this.target, this.handle);
    withParameters(this.gl, parameters, () => {
      // TODO - x,y parameters
      if (compressed) {
        this.gl.compressedTexSubImage2D(target, level, x, y, width, height, format, data);
      } else if (data === null) {
        this.gl.texSubImage2D(target, level, format, width, height, border, dataFormat, type, null);
      } else if (ArrayBuffer.isView(data)) {
        this.gl.texSubImage2D(target, level, x, y, width, height, format, type, data, offset);
      } else if (data instanceof WebGLBuffer) {
        // WebGL2 allows us to create texture directly from a WebGL buffer
        assertWebGL2Context(this.gl); // This texImage2D signature uses currently bound GL_PIXEL_UNPACK_BUFFER

        this.gl.bindBuffer(GL.PIXEL_UNPACK_BUFFER, data);
        this.gl.texSubImage2D(target, level, format, width, height, border, format, type, offset);
        this.gl.bindBuffer(GL.GL_PIXEL_UNPACK_BUFFER, null);
      } else {
        // Assume data is a browser supported object (ImageData, Canvas, ...)
        this.gl.texSubImage2D(target, level, x, y, format, type, data);
      }
    });
    this.gl.bindTexture(this.target, null);
  }
  /* eslint-enable max-len, max-statements, complexity */

  /**
   * Defines a two-dimensional texture image or cube-map texture image with
   * pixels from the current framebuffer (rather than from client memory).
   * (gl.copyTexImage2D wrapper)
   *
   * Note that binding a texture into a Framebuffer's color buffer and
   * rendering can be faster.
   */


  copyFramebuffer({
    target = this.target,
    framebuffer,
    offset = 0,
    x = 0,
    y = 0,
    width,
    height,
    level = 0,
    internalFormat = GL.RGBA,
    border = 0
  }) {
    if (framebuffer) {
      framebuffer.bind();
    } // target


    this.bind();
    this.gl.copyTexImage2D(this.target, level, internalFormat, x, y, width, height, border);
    this.unbind();

    if (framebuffer) {
      framebuffer.unbind();
    }
  }

  getActiveUnit() {
    return this.gl.getParameter(GL.ACTIVE_TEXTURE) - GL.TEXTURE0;
  } // target cannot be modified by bind:
  // textures are special because when you first bind them to a target,
  // they get special information. When you first bind a texture as a
  // GL_TEXTURE_2D, you are actually setting special state in the texture.
  // You are saying that this texture is a 2D texture.
  // And it will always be a 2D texture; this state cannot be changed ever.
  // If you have a texture that was first bound as a GL_TEXTURE_2D,
  // you must always bind it as a GL_TEXTURE_2D;
  // attempting to bind it as GL_TEXTURE_1D will give rise to an error
  // (while run-time).


  bind(textureUnit = this.textureUnit) {
    if (textureUnit === undefined) {
      throw new Error('Texture.bind: must specify texture unit');
    }

    this.textureUnit = textureUnit;
    this.gl.activeTexture(GL.TEXTURE0 + textureUnit);
    this.gl.bindTexture(this.target, this.handle);
    return textureUnit;
  }

  unbind() {
    if (this.textureUnit === undefined) {
      throw new Error('Texture.unbind: texture unit not specified');
    }

    this.gl.activeTexture(GL.TEXTURE0 + this.textureUnit);
    this.gl.bindTexture(this.target, null);
    return this.textureUnit;
  } // PRIVATE METHODS


  _getDataType({
    data,
    compressed = false
  }) {
    if (compressed) {
      return {
        data,
        dataType: 'compressed'
      };
    }

    if (data === null) {
      return {
        data,
        dataType: 'null'
      };
    }

    if (ArrayBuffer.isView(data)) {
      return {
        data,
        dataType: 'typed-array'
      };
    }

    if (data instanceof Buffer) {
      return {
        data: data.handle,
        dataType: 'buffer'
      };
    }

    if (data instanceof WebGLBuffer) {
      return {
        data,
        dataType: 'buffer'
      };
    } // Assume data is a browser supported object (ImageData, Canvas, ...)


    return {
      data,
      dataType: 'browser-object'
    };
  } // Image 3D copies from Typed Array or WebGLBuffer


  setImage3D({
    level = 0,
    internalformat = GL.RGBA,
    width,
    height,
    depth = 1,
    border = 0,
    format,
    type = GL.UNSIGNED_BYTE,
    offset = 0,
    pixels
  }) {
    if (ArrayBuffer.isView(pixels)) {
      this.gl.texImage3D(this.target, level, internalformat, width, height, depth, border, format, type, pixels);
      return this;
    }

    if (pixels instanceof Buffer) {
      this.gl.bindBuffer(GL.PIXEL_UNPACK_BUFFER, pixels.handle);
      this.gl.texImage3D(this.target, level, internalformat, width, height, depth, border, format, type, offset);
    }

    return this;
  }
  /* Copied from texture-2d.js
  // WebGL2
  setPixels(opts = {}) {
    const {
      buffer,
      width = null,
      height = null,
      mipmapLevel = 0,
      format = GL.RGBA,
      type = GL.UNSIGNED_BYTE,
      border = 0
    } = opts;
     const {gl} = this;
     // This signature of texImage2D uses currently bound GL_PIXEL_UNPACK_BUFFER
    gl.bindBuffer(GL.PIXEL_UNPACK_BUFFER, buffer.target);
    // And as always, we must also bind the texture itself
    this.bind();
     gl.texImage2D(gl.TEXTURE_2D,
      mipmapLevel, format, width, height, border, format, type, buffer.target);
     this.unbind();
    gl.bindBuffer(GL.GL_PIXEL_UNPACK_BUFFER, null);
    return this;
  }
   setImageDataFromCompressedBuffer(opts) {
    const {
      buffer,
      // offset = 0,
      width = null,
      height = null,
      mipmapLevel = 0,
      internalFormat = GL.RGBA,
      // format = GL.RGBA,
      // type = GL.UNSIGNED_BYTE,
      border = 0
    } = opts;
     const {gl} = this;
    gl.compressedTexImage2D(this.target,
      mipmapLevel, internalFormat, width, height, border, buffer);
    // gl.compressedTexSubImage2D(target,
    //   level, xoffset, yoffset, width, height, format, ArrayBufferView? pixels);
    return this;
  }
   copySubImage(opts) {
    const {
      // pixels,
      // offset = 0,
      // x,
      // y,
      // width,
      // height,
      // mipmapLevel = 0,
      // internalFormat = GL.RGBA,
      // type = GL.UNSIGNED_BYTE,
      // border = 0
    } = opts;
     // if (pixels instanceof ArrayBufferView) {
    //   gl.texSubImage2D(target, level, x, y, width, height, format, type, pixels);
    // }
    // gl.texSubImage2D(target, level, x, y, format, type, ? pixels);
    // gl.texSubImage2D(target, level, x, y, format, type, HTMLImageElement pixels);
    // gl.texSubImage2D(target, level, x, y, format, type, HTMLCanvasElement pixels);
    // gl.texSubImage2D(target, level, x, y, format, type, HTMLVideoElement pixels);
    // // Additional signature in a WebGL 2 context:
    // gl.texSubImage2D(target, level, x, y, format, type, GLintptr offset);
  }
  */
  // HELPER METHODS


  _deduceParameters(opts) {
    const format = opts.format,
          data = opts.data;
    let width = opts.width,
        height = opts.height,
        dataFormat = opts.dataFormat,
        type = opts.type,
        compressed = opts.compressed; // Deduce format and type from format

    const textureFormat = TEXTURE_FORMATS[format];
    dataFormat = dataFormat || textureFormat && textureFormat.dataFormat;
    type = type || textureFormat && textureFormat.types[0]; // Deduce compression from format

    compressed = compressed || textureFormat && textureFormat.compressed;

    var _deduceImageSize = this._deduceImageSize({
      data,
      width,
      height
    });

    width = _deduceImageSize.width;
    height = _deduceImageSize.height;
    return {
      dataFormat,
      type,
      compressed,
      width,
      height,
      format,
      data
    };
  }
  /* global ImageData, HTMLImageElement, HTMLCanvasElement, HTMLVideoElement */


  _deduceImageSize({
    data,
    width,
    height
  }) {
    let size;

    if (typeof ImageData !== 'undefined' && data instanceof ImageData) {
      size = {
        width: data.width,
        height: data.height
      };
    } else if (typeof HTMLImageElement !== 'undefined' && data instanceof HTMLImageElement) {
      size = {
        width: data.naturalWidth,
        height: data.naturalHeight
      };
    } else if (typeof HTMLCanvasElement !== 'undefined' && data instanceof HTMLCanvasElement) {
      size = {
        width: data.width,
        height: data.height
      };
    } else if (typeof HTMLVideoElement !== 'undefined' && data instanceof HTMLVideoElement) {
      size = {
        width: data.videoWidth,
        height: data.videoHeight
      };
    } else if (!data) {
      size = {
        width: width >= 0 ? width : 1,
        height: height >= 0 ? height : 1
      };
    } else {
      size = {
        width,
        height
      };
    }

    assert(size, 'Could not deduced texture size');
    assert(width === undefined || size.width === width, 'Deduced texture width does not match supplied width');
    assert(height === undefined || size.height === height, 'Deduced texture height does not match supplied height');
    return size;
  } // RESOURCE METHODS


  _createHandle() {
    return this.gl.createTexture();
  }

  _deleteHandle() {
    this.gl.deleteTexture(this.handle);
  }

  _getParameter(pname) {
    switch (pname) {
      case GL.TEXTURE_WIDTH:
        return this.width;

      case GL.TEXTURE_HEIGHT:
        return this.height;

      default:
        this.gl.bindTexture(this.target, this.handle);
        const value = this.gl.getTexParameter(this.target, pname);
        this.gl.bindTexture(this.target, null);
        return value;
    }
  }

  _setParameter(pname, param) {
    this.gl.bindTexture(this.target, this.handle); // NOTE: Apply NPOT workaround

    param = this._getNPOTParam(pname, param); // Apparently there are some integer/float conversion rules that made
    // the WebGL committe expose two parameter setting functions in JavaScript.
    // For now, pick the float version for parameters specified as GLfloat.

    switch (pname) {
      case GL.TEXTURE_MIN_LOD:
      case GL.TEXTURE_MAX_LOD:
        this.gl.texParameterf(this.handle, pname, param);
        break;

      case GL.TEXTURE_WIDTH:
      case GL.TEXTURE_HEIGHT:
        throw new Error('Cannot set emulated parameter');

      default:
        this.gl.texParameteri(this.target, pname, param);
        break;
    }

    this.gl.bindTexture(this.target, null);
    return this;
  }

  _isNPOT() {
    return !isWebGL2(this.gl) && (!isPowerOfTwo(this.width) || !isPowerOfTwo(this.height));
  } // Update default settings which are not supported by NPOT textures.


  _updateForNPOT(parameters) {
    if (parameters[this.gl.TEXTURE_MIN_FILTER] === undefined) {
      log.warn(`texture: ${this} is Non-Power-Of-Two, forcing TEXTURE_MIN_FILTER to LINEAR`)();
      parameters[this.gl.TEXTURE_MIN_FILTER] = this.gl.LINEAR;
    }

    if (parameters[this.gl.TEXTURE_WRAP_S] === undefined) {
      log.warn(`texture: ${this} is Non-Power-Of-Two, forcing TEXTURE_WRAP_S to CLAMP_TO_EDGE`)();
      parameters[this.gl.TEXTURE_WRAP_S] = this.gl.CLAMP_TO_EDGE;
    }

    if (parameters[this.gl.TEXTURE_WRAP_T] === undefined) {
      log.warn(`texture: ${this} is Non-Power-Of-Two, forcing TEXTURE_WRAP_T to CLAMP_TO_EDGE`)();
      parameters[this.gl.TEXTURE_WRAP_T] = this.gl.CLAMP_TO_EDGE;
    }
  }

  _getNPOTParam(pname, param) {
    if (this._isNPOT()) {
      switch (pname) {
        case GL.TEXTURE_MIN_FILTER:
          if (NPOT_MIN_FILTERS.indexOf(param) === -1) {
            log.warn(`texture: ${this} is Non-Power-Of-Two, forcing TEXTURE_MIN_FILTER to LINEAR`)();
            param = GL.LINEAR;
          }

          break;

        case GL.TEXTURE_WRAP_S:
        case GL.TEXTURE_WRAP_T:
          if (param !== GL.CLAMP_TO_EDGE) {
            log.warn(`texture: ${this} is Non-Power-Of-Two, ${getKey(this.gl, pname)} to CLAMP_TO_EDGE`)();
            param = GL.CLAMP_TO_EDGE;
          }

          break;

        default:
          break;
      }
    }

    return param;
  }

}
//# sourceMappingURL=texture.js.map