import assert from '../../utils/assert';

const shaderModules = {};
let defaultShaderModules = [];

/**
 * Registers an array of shader modules
 * @param {Object[]} shaderModuleList - Array of shader modules
 */
export function registerShaderModules(shaderModuleList, {
  ignoreMultipleRegistrations = false
} = {}) {
  for (const shaderModule of shaderModuleList) {
    registerShaderModule(shaderModule, {ignoreMultipleRegistrations});
  }
}

/**
 * Registers an array of default shader modules. These will be concatenated
 * automatically at the end of any shader module list passed to
 * `assembleShaders` (plus `resolveModules` and `getShaderDependencies`)
 * @param {Object[]} modules - Array of shader modules
 */
export function setDefaultShaderModules(modules) {
  defaultShaderModules = modules;
}

// Looks up a moduleName among registered modules and returns definition.
// If "inline" module, returns it directly
export function getShaderModule(moduleOrName) {
  // Check if "inline" module
  if (typeof moduleOrName !== 'string') {
    const shaderModule = moduleOrName;
    assert(typeof shaderModule.name === 'string');
    registerShaderModule(shaderModule, {ignoreMultipleRegistrations: true});
    return shaderModule;
  }

  // Look up module
  const shaderModule = shaderModules[moduleOrName];
  if (!shaderModule) {
    // console.log(`${moduleOrName} not in registered modules:`, shaderModules);
    assert(false, `Unknown shader module ${moduleOrName}`);
  }
  return shaderModule;
}

// registers any supplied modules and returns a list of module names
export function resolveModules(modules) {
  const moduleNames = modules.map(module => {
    if (typeof module !== 'string') {
      registerShaderModules([module], {ignoreMultipleRegistrations: true});
      return module.name;
    }
    return module;
  });

  return getShaderDependencies(moduleNames);
}

/**
 * Takes a list of shader module names and returns a new list of
 * shader module names that includes all dependencies, sorted so
 * that modules that are dependencies of other modules come first.
 *
 * If the shader glsl code from the returned modules is concatenated
 * in the reverse order, it is guaranteed that all functions be resolved and
 * that all function and variable definitions come before use.
 *
 * @param {String[]} modules - Array of modules (inline modules or module names)
 * @return {String[]} - Array of modules
 */
export function getShaderDependencies(modules) {
  modules = modules.concat(defaultShaderModules);

  const result = {};
  getDependencyGraph({
    modules,
    level: 0,
    result
  });

  // Return a reverse sort so that dependencies come before the modules that use them
  return Object.keys(result).sort((a, b) => result[b] - result[a]);
}

/**
 * Recursively checks module dpendencies to calculate dependency
 * level of each module.
 *
 * @param {String[]} modules - Array of modules
 * @param {Number} level - Current level
 * @return {result} - Map of module name to its level
 */
// Adds another level of dependencies to the result map
export function getDependencyGraph({modules, level, result}) {
  if (level >= 5) {
    throw new Error('Possible loop in shader dependency graph');
  }

  // Update level on all current modules
  for (const moduleOrName of modules) {
    const shaderModule = getShaderModule(moduleOrName);
    if (result[shaderModule.name] === undefined || result[shaderModule.name] < level) {
      result[shaderModule.name] = level;
    }
  }

  // Recurse
  for (const moduleOrName of modules) {
    const shaderModule = getShaderModule(moduleOrName);

    if (shaderModule.dependencies) {
      getDependencyGraph({
        modules: shaderModule.dependencies,
        level: level + 1,
        result
      });
    }
  }

  return result;
}

// PRIVATE API

function parseDeprecationDefinitions(deprecations = []) {
  deprecations.forEach(def => {
    switch (def.type) {
    case 'function':
      def.regex = new RegExp(`\\b${def.old}\\(`);
      break;
    default:
      def.regex = new RegExp(`${def.type} ${def.old};`);
    }
  });

  return deprecations;
}

function registerShaderModule(shaderModule, {ignoreMultipleRegistrations = false}) {
  assert(shaderModule.name, 'shader module has no name');
  if (!ignoreMultipleRegistrations && shaderModules[shaderModule.name]) {
    // TODO - instead verify that definition is not changing...
    throw new Error(`shader module ${shaderModule.name} already registered`);
  }
  shaderModules[shaderModule.name] = shaderModule;
  shaderModule.dependencies = shaderModule.dependencies || [];
  shaderModule.deprecations = parseDeprecationDefinitions(shaderModule.deprecations);
}
