/* eslint-disable no-inline-comments */
import GL from '../constants';
import {isWebGL2} from '../webgl-utils';
import Resource from './resource';
import assert from '../utils/assert';

// Define local GL constants to optimize minification
const GL_RENDERBUFFER = 0x8D41;
const GL_SAMPLES = 0x80A9;
const GL_RENDERBUFFER_WIDTH = 0x8D42;
const GL_RENDERBUFFER_HEIGHT = 0x8D43;
const GL_RENDERBUFFER_INTERNAL_FORMAT = 0x8D44;
const GL_RENDERBUFFER_SAMPLES = 0x8CAB;

// Define local extension strings to optimize minification
// const SRGB = 'EXT_sRGB';
// const CB_FLOAT_WEBGL1 = 'WEBGL_color_buffer_float';
const CB_FLOAT_WEBGL2 = 'EXT_color_buffer_float';
// const CB_HALF_FLOAT_WEBGL1 = 'EXT_color_buffer_half_float';

export const RENDERBUFFER_FORMATS = {
  [GL.DEPTH_COMPONENT16]: {}, // 16 depth bits.
  [GL.DEPTH_COMPONENT24]: {gl2: true},
  [GL.DEPTH_COMPONENT32F]: {gl2: true},

  [GL.STENCIL_INDEX8]: {}, // 8 stencil bits.

  [GL.DEPTH_STENCIL]: {},
  [GL.DEPTH24_STENCIL8]: {gl2: true},
  [GL.DEPTH32F_STENCIL8]: {gl2: true},

  // When using a WebGL 1 context, color renderbuffer formats are limited
  [GL.RGBA4]: {},
  [GL.RGB565]: {},
  [GL.RGB5_A1]: {},

  // When using a WebGL 2 context, the following values are available additionally:
  [GL.R8]: {gl2: true},
  [GL.R8UI]: {gl2: true},
  [GL.R8I]: {gl2: true},
  [GL.R16UI]: {gl2: true},
  [GL.R16I]: {gl2: true},
  [GL.R32UI]: {gl2: true},
  [GL.R32I]: {gl2: true},
  [GL.RG8]: {gl2: true},
  [GL.RG8UI]: {gl2: true},
  [GL.RG8I]: {gl2: true},
  [GL.RG16UI]: {gl2: true},
  [GL.RG16I]: {gl2: true},
  [GL.RG32UI]: {gl2: true},
  [GL.RG32I]: {gl2: true},
  [GL.RGB8]: {gl2: true},
  [GL.RGBA8]: {gl2: true},
  // [GL.SRGB8_ALPHA8]: {gl2: true, gl1: SRGB}, // When using the EXT_sRGB WebGL1 extension
  [GL.RGB10_A2]: {gl2: true},
  [GL.RGBA8UI]: {gl2: true},
  [GL.RGBA8I]: {gl2: true},
  [GL.RGB10_A2UI]: {gl2: true},
  [GL.RGBA16UI]: {gl2: true},
  [GL.RGBA16I]: {gl2: true},
  [GL.RGBA32I]: {gl2: true},
  [GL.RGBA32UI]: {gl2: true},

  // When using a WebGL 2 context and the EXT_color_buffer_float WebGL2 extension
  [GL.R16F]: {gl2: CB_FLOAT_WEBGL2},
  [GL.RG16F]: {gl2: CB_FLOAT_WEBGL2},
  [GL.RGBA16F]: {gl2: CB_FLOAT_WEBGL2},
  [GL.R32F]: {gl2: CB_FLOAT_WEBGL2},
  [GL.RG32F]: {gl2: CB_FLOAT_WEBGL2},
  // TODO - can't get WEBGL_color_buffer_float to work on renderbuffers
  [GL.RGBA32F]: {gl2: CB_FLOAT_WEBGL2},
  // [GL.RGBA32F]: {gl2: CB_FLOAT_WEBGL2, gl1: CB_FLOAT_WEBGL1},
  [GL.R11F_G11F_B10F]: {gl2: CB_FLOAT_WEBGL2}
};

function isFormatSupported(gl, format, formats) {
  const info = formats[format];
  if (!info) {
    return false;
  }
  const value = isWebGL2(gl) ? info.gl2 || info.gl1 : info.gl1;
  if (typeof value === 'string') {
    return gl.getExtension(value);
  }
  return value;
}

export default class Renderbuffer extends Resource {

  static isSupported(gl, {format} = {}) {
    return !format || isFormatSupported(gl, format, RENDERBUFFER_FORMATS);
  }

  static getSamplesForFormat(gl, {format}) {
    // Polyfilled to return [0] under WebGL1
    return gl.getInternalformatParameter(GL_RENDERBUFFER, format, GL_SAMPLES);
  }

  constructor(gl, opts = {}) {
    super(gl, opts);
    this.initialize(opts);
    Object.seal(this);
  }

  // Creates and initializes a renderbuffer object's data store
  initialize({format, width = 1, height = 1, samples = 0}) {
    assert(format, 'Needs format');
    this.gl.bindRenderbuffer(GL_RENDERBUFFER, this.handle);

    if (samples !== 0 && isWebGL2(this.gl)) {
      this.gl.renderbufferStorageMultisample(GL_RENDERBUFFER, samples, format, width, height);
    } else {
      this.gl.renderbufferStorage(GL_RENDERBUFFER, format, width, height);
    }

    // this.gl.bindRenderbuffer(GL_RENDERBUFFER, null);

    this.format = format;
    this.width = width;
    this.height = height;
    this.samples = samples;

    return this;
  }

  resize({width, height}) {
    // Don't resize if width/height haven't changed
    if (width !== this.width || height !== this.height) {
      return this.initialize({width, height, format: this.format, samples: this.samples});
    }
    return this;
  }

  // PRIVATE METHODS
  _createHandle() {
    return this.gl.createRenderbuffer();
  }

  _deleteHandle() {
    this.gl.deleteRenderbuffer(this.handle);
  }

  _syncHandle(handle) {
    this.format = this.getParameter(GL_RENDERBUFFER_INTERNAL_FORMAT);
    this.width = this.getParameter(GL_RENDERBUFFER_WIDTH);
    this.height = this.getParameter(GL_RENDERBUFFER_HEIGHT);
    this.samples = this.getParameter(GL_RENDERBUFFER_SAMPLES);
  }

  // @param {Boolean} opt.autobind=true - method call will bind/unbind object
  // @returns {GLenum|GLint} - depends on pname
  _getParameter(pname) {
    this.gl.bindRenderbuffer(GL_RENDERBUFFER, this.handle);
    const value = this.gl.getRenderbufferParameter(GL_RENDERBUFFER, pname);
    // this.gl.bindRenderbuffer(GL_RENDERBUFFER, null);
    return value;
  }
}
