'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configure = configure;
exports.checkNumber = checkNumber;
exports.formatValue = formatValue;
exports.formatAngle = formatAngle;
exports.isArray = isArray;
exports.clone = clone;
exports.radians = radians;
exports.degrees = degrees;
exports.sin = sin;
exports.cos = cos;
exports.tan = tan;
exports.asin = asin;
exports.acos = acos;
exports.atan = atan;
exports.clamp = clamp;
exports.equals = equals;
// Copyright (c) 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* eslint-disable no-shadow */
// TODO - remove
var config = {};
config.EPSILON = 1e-12;
config.debug = true;
config.precision = 4;
config.printTypes = false;
config.printDegrees = false;
config.printRowMajor = true;

exports.config = config;
function configure(options) {
  if ('epsilon' in options) {
    config.EPSILON = options.epsilon;
  }

  if ('debug' in options) {
    config.debug = options.debug;
  }
}

function checkNumber(value) {
  if (!Number.isFinite(value)) {
    throw new Error('Invalid number ' + value);
  }
  return value;
}

function round(value) {
  return Math.round(value / config.EPSILON) * config.EPSILON;
}

function formatValue(value) {
  var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref$precision = _ref.precision,
      precision = _ref$precision === undefined ? config.precision || 4 : _ref$precision;

  value = round(value);
  return parseFloat(value.toPrecision(precision));
}

function formatAngle(value) {
  var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref2$precision = _ref2.precision,
      precision = _ref2$precision === undefined ? config.precision || 4 : _ref2$precision,
      _ref2$printDegrees = _ref2.printDegrees,
      printDegrees = _ref2$printDegrees === undefined ? config.printAngles : _ref2$printDegrees;

  value = printDegrees ? degrees(value) : value;
  value = round(value);
  return '' + parseFloat(value.toPrecision(precision)) + (printDegrees ? '°' : '');
}

// Returns true if value is either an array or a typed array
// Note: does not return true for ArrayBuffers and DataViews
function isArray(value) {
  return Array.isArray(value) || ArrayBuffer.isView(value) && value.length !== undefined;
}

// If the array has a clone function, calls it, otherwise returns a copy
function clone(array) {
  return array.clone ? array.clone() : new Array(array);
}

// If the argument value is an array, applies the func element wise,
// otherwise applies func to the argument value
function map(value, func) {
  if (isArray(value)) {
    var result = clone(value);
    for (var i = 0; i < result.length; ++i) {
      result[i] = func(result[i], i, result);
    }
    return result;
  }
  return func(value);
}

//
// GLSL math function equivalents
// Works on both single values and vectors
//

function radians(degrees) {
  return map(degrees, function (degrees) {
    return degrees / 180 * Math.PI;
  });
}

// GLSL equivalent: Works on single values and vectors
function degrees(radians) {
  return map(radians, function (radians) {
    return radians * 180 / Math.PI;
  });
}

// GLSL equivalent: Works on single values and vectors
function sin(radians) {
  return map(radians, function (angle) {
    return Math.sin(angle);
  });
}

// GLSL equivalent: Works on single values and vectors
function cos(radians) {
  return map(radians, function (angle) {
    return Math.cos(angle);
  });
}

// GLSL equivalent: Works on single values and vectors
function tan(radians) {
  return map(radians, function (angle) {
    return Math.tan(angle);
  });
}

// GLSL equivalent: Works on single values and vectors
function asin(radians) {
  return map(radians, function (angle) {
    return Math.asin(angle);
  });
}

// GLSL equivalent: Works on single values and vectors
function acos(radians) {
  return map(radians, function (angle) {
    return Math.acos(angle);
  });
}

// GLSL equivalent: Works on single values and vectors
function atan(radians) {
  return map(radians, function (angle) {
    return Math.atan(angle);
  });
}

function clamp(value, min, max) {
  return map(value, function (value) {
    return Math.max(min, Math.min(max, value));
  });
}

function equals(a, b) {
  if (Array.isArray(a) && Array.isArray(b)) {
    if (a === b) {
      return true;
    }
    if (a.length !== b.length) {
      return false;
    }
    for (var i = 0; i < a.length; ++i) {
      if (!equals(a[i], b[i])) {
        return false;
      }
    }
    return true;
  }
  return Math.abs(a - b) <= config.EPSILON * Math.max(1.0, Math.abs(a), Math.abs(b));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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