"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configure = configure;
exports.checkNumber = checkNumber;
exports.formatValue = formatValue;
exports.formatAngle = formatAngle;
exports.isArray = isArray;
exports.clone = clone;
exports.radians = radians;
exports.degrees = degrees;
exports.sin = sin;
exports.cos = cos;
exports.tan = tan;
exports.asin = asin;
exports.acos = acos;
exports.atan = atan;
exports.clamp = clamp;
exports.lerp = lerp;
exports.equals = equals;
exports.config = void 0;
// Copyright (c) 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

/* eslint-disable no-shadow */
// TODO - remove
var config = {};
exports.config = config;
config.EPSILON = 1e-12;
config.debug = true;
config.precision = 4;
config.printTypes = false;
config.printDegrees = false;
config.printRowMajor = true;

function configure(options) {
  if ('epsilon' in options) {
    config.EPSILON = options.epsilon;
  }

  if ('debug' in options) {
    config.debug = options.debug;
  }
}

function checkNumber(value) {
  if (!Number.isFinite(value)) {
    throw new Error("Invalid number ".concat(value));
  }

  return value;
}

function round(value) {
  return Math.round(value / config.EPSILON) * config.EPSILON;
}

function formatValue(value) {
  var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref$precision = _ref.precision,
      precision = _ref$precision === void 0 ? config.precision || 4 : _ref$precision;

  value = round(value);
  return parseFloat(value.toPrecision(precision));
}

function formatAngle(value) {
  var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref2$precision = _ref2.precision,
      precision = _ref2$precision === void 0 ? config.precision || 4 : _ref2$precision,
      _ref2$printDegrees = _ref2.printDegrees,
      printDegrees = _ref2$printDegrees === void 0 ? config.printAngles : _ref2$printDegrees;

  value = printDegrees ? degrees(value) : value;
  value = round(value);
  return "".concat(parseFloat(value.toPrecision(precision))).concat(printDegrees ? '°' : '');
} // Returns true if value is either an array or a typed array
// Note: does not return true for ArrayBuffers and DataViews


function isArray(value) {
  return Array.isArray(value) || ArrayBuffer.isView(value) && value.length !== undefined;
} // If the array has a clone function, calls it, otherwise returns a copy


function clone(array) {
  return array.clone ? array.clone() : new Array(array);
} // If the argument value is an array, applies the func element wise,
// otherwise applies func to the argument value


function map(value, func) {
  if (isArray(value)) {
    var result = clone(value);

    for (var i = 0; i < result.length; ++i) {
      result[i] = func(result[i], i, result);
    }

    return result;
  }

  return func(value);
} //
// GLSL math function equivalents
// Works on both single values and vectors
//


function radians(degrees) {
  return map(degrees, function (degrees) {
    return degrees / 180 * Math.PI;
  });
} // GLSL equivalent: Works on single values and vectors


function degrees(radians) {
  return map(radians, function (radians) {
    return radians * 180 / Math.PI;
  });
} // GLSL equivalent: Works on single values and vectors


function sin(radians) {
  return map(radians, function (angle) {
    return Math.sin(angle);
  });
} // GLSL equivalent: Works on single values and vectors


function cos(radians) {
  return map(radians, function (angle) {
    return Math.cos(angle);
  });
} // GLSL equivalent: Works on single values and vectors


function tan(radians) {
  return map(radians, function (angle) {
    return Math.tan(angle);
  });
} // GLSL equivalent: Works on single values and vectors


function asin(radians) {
  return map(radians, function (angle) {
    return Math.asin(angle);
  });
} // GLSL equivalent: Works on single values and vectors


function acos(radians) {
  return map(radians, function (angle) {
    return Math.acos(angle);
  });
} // GLSL equivalent: Works on single values and vectors


function atan(radians) {
  return map(radians, function (angle) {
    return Math.atan(angle);
  });
}

function clamp(value, min, max) {
  return map(value, function (value) {
    return Math.max(min, Math.min(max, value));
  });
} // Interpolate between two numbers or two arrays


function lerp(a, b, t) {
  if (Array.isArray(a)) {
    return a.map(function (ai, i) {
      return lerp(ai, b[i], t);
    });
  }

  return t * b + (1 - t) * a;
}

function equals(a, b) {
  if (Array.isArray(a) && Array.isArray(b)) {
    if (a === b) {
      return true;
    }

    if (a.length !== b.length) {
      return false;
    }

    for (var i = 0; i < a.length; ++i) {
      if (!equals(a[i], b[i])) {
        return false;
      }
    }

    return true;
  }

  return Math.abs(a - b) <= config.EPSILON * Math.max(1.0, Math.abs(a), Math.abs(b));
}
//# sourceMappingURL=common.js.map