import MathArray from './lib/math-array';
import { validateVector } from './lib/validators';
import { checkNumber } from './lib/common';
import Vector2 from './vector2';
import Vector3 from './vector3';
import * as mat3 from 'gl-matrix/mat3';
import * as vec2 from 'gl-matrix/vec2';
import * as vec3 from 'gl-matrix/vec3';
const IDENTITY = [1, 0, 0, 0, 1, 0, 0, 0, 1];
export function validateMatrix3(m) {
  return m.length === 9 && Number.isFinite(m[0]) && Number.isFinite(m[1]) && Number.isFinite(m[2]) && Number.isFinite(m[3]) && Number.isFinite(m[4]) && Number.isFinite(m[5]) && Number.isFinite(m[6]) && Number.isFinite(m[7]) && Number.isFinite(m[8]);
}
export default class Matrix3 extends MathArray {
  constructor() {
    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    super();

    if (Array.isArray(args[0]) && arguments.length === 1) {
      this.copy(args[0]);
    } else {
      this.identity();
    }
  }

  get ELEMENTS() {
    return 9;
  }

  setRowMajor() {
    let m00 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
    let m01 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let m02 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    let m10 = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
    let m11 = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : 1;
    let m12 = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 0;
    let m20 = arguments.length > 6 && arguments[6] !== undefined ? arguments[6] : 0;
    let m21 = arguments.length > 7 && arguments[7] !== undefined ? arguments[7] : 0;
    let m22 = arguments.length > 8 && arguments[8] !== undefined ? arguments[8] : 1;
    this[0] = m00;
    this[1] = m10;
    this[2] = m20;
    this[3] = m01;
    this[4] = m11;
    this[5] = m21;
    this[6] = m02;
    this[7] = m12;
    this[8] = m22;
    return this.check();
  }

  setColumnMajor() {
    let m00 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
    let m10 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let m20 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    let m01 = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
    let m11 = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : 1;
    let m21 = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 0;
    let m02 = arguments.length > 6 && arguments[6] !== undefined ? arguments[6] : 0;
    let m12 = arguments.length > 7 && arguments[7] !== undefined ? arguments[7] : 0;
    let m22 = arguments.length > 8 && arguments[8] !== undefined ? arguments[8] : 1;
    this[0] = m00;
    this[1] = m10;
    this[2] = m20;
    this[3] = m01;
    this[4] = m11;
    this[5] = m21;
    this[6] = m02;
    this[7] = m12;
    this[8] = m22;
    return this.check();
  }

  copy(array) {
    return this.setColumnMajor(...array);
  }

  set() {
    return this.setColumnMajor(...arguments);
  }

  getElement(i, j) {
    let columnMajor = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
    return columnMajor ? this[i * 3 + j] : this[j * 3 + i];
  }

  setElement(i, j, value) {
    let columnMajor = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

    if (columnMajor) {
      this[i * 3 + j] = checkNumber(value);
    } else {
      this[j * 3 + i] = checkNumber(value);
    }

    return this;
  }

  determinant() {
    return mat3.determinant(this);
  }

  identity() {
    for (let i = 0; i < IDENTITY.length; ++i) {
      this[i] = IDENTITY[i];
    }

    return this.check();
  }

  fromQuaternion(q) {
    mat3.fromQuat(this, q);
    return this.check();
  }

  transpose() {
    mat3.transpose(this, this);
    return this.check();
  }

  invert() {
    mat3.invert(this, this);
    return this.check();
  }

  multiplyLeft(a) {
    mat3.multiply(this, a, this);
    return this.check();
  }

  multiplyRight(a) {
    mat3.multiply(this, this, a);
    return this.check();
  }

  rotate(radians) {
    mat3.rotate(this, this, radians);
    return this.check();
  }

  scale(factor) {
    if (Array.isArray(factor)) {
      mat3.scale(this, this, factor);
    } else {
      mat3.scale(this, this, [factor, factor, factor]);
    }

    return this.check();
  }

  translate(vec) {
    mat3.translate(this, this, vec);
    return this.check();
  }

  transformVector2(vector, out) {
    out = out || new Vector2();
    vec2.transformMat3(out, vector, this);
    validateVector(out, 2);
    return out;
  }

  transformVector3(vector, out) {
    out = out || new Vector3();
    vec3.transformMat3(out, vector, this);
    validateVector(out, 3);
    return out;
  }

  transformVector(vector, out) {
    switch (vector.length) {
      case 2:
        return this.transformVector2(vector, out);

      case 3:
        return this.transformVector3(vector, out);

      default:
        throw new Error('Illegal vector');
    }
  }

}
//# sourceMappingURL=matrix3.js.map