import _slicedToArray from "@babel/runtime/helpers/esm/slicedToArray";
import MathArray from './lib/math-array';
import { checkNumber } from './lib/common';
import { validateVector } from './lib/validators';
import Vector2 from './vector2';
import Vector3 from './vector3';
import Vector4 from './vector4';
import * as mat4 from 'gl-matrix/mat4';
import * as vec2 from 'gl-matrix/vec2';
import * as vec3 from 'gl-matrix/vec3';
import * as vec4 from 'gl-matrix/vec4';
const IDENTITY = [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1];
export function validateMatrix4(m) {
  return m.length === 16 && Number.isFinite(m[0]) && Number.isFinite(m[1]) && Number.isFinite(m[2]) && Number.isFinite(m[3]) && Number.isFinite(m[4]) && Number.isFinite(m[5]) && Number.isFinite(m[6]) && Number.isFinite(m[7]) && Number.isFinite(m[8]) && Number.isFinite(m[9]) && Number.isFinite(m[10]) && Number.isFinite(m[11]) && Number.isFinite(m[12]) && Number.isFinite(m[13]) && Number.isFinite(m[14]) && Number.isFinite(m[15]);
}
export default class Matrix4 extends MathArray {
  constructor() {
    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    super();

    if (Array.isArray(args[0]) && arguments.length === 1) {
      this.copy(args[0]);
    } else {
      this.identity();
    }
  }

  get ELEMENTS() {
    return 16;
  }

  setRowMajor() {
    let m00 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
    let m01 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let m02 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    let m03 = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
    let m10 = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : 0;
    let m11 = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 1;
    let m12 = arguments.length > 6 && arguments[6] !== undefined ? arguments[6] : 0;
    let m13 = arguments.length > 7 && arguments[7] !== undefined ? arguments[7] : 0;
    let m20 = arguments.length > 8 && arguments[8] !== undefined ? arguments[8] : 0;
    let m21 = arguments.length > 9 && arguments[9] !== undefined ? arguments[9] : 0;
    let m22 = arguments.length > 10 && arguments[10] !== undefined ? arguments[10] : 1;
    let m23 = arguments.length > 11 && arguments[11] !== undefined ? arguments[11] : 0;
    let m30 = arguments.length > 12 && arguments[12] !== undefined ? arguments[12] : 0;
    let m31 = arguments.length > 13 && arguments[13] !== undefined ? arguments[13] : 0;
    let m32 = arguments.length > 14 && arguments[14] !== undefined ? arguments[14] : 0;
    let m33 = arguments.length > 15 && arguments[15] !== undefined ? arguments[15] : 1;
    this[0] = m00;
    this[1] = m10;
    this[2] = m20;
    this[3] = m30;
    this[4] = m01;
    this[5] = m11;
    this[6] = m21;
    this[7] = m31;
    this[8] = m02;
    this[9] = m12;
    this[10] = m22;
    this[11] = m32;
    this[12] = m03;
    this[13] = m13;
    this[14] = m23;
    this[15] = m33;
    return this.check();
  }

  setColumnMajor() {
    let m00 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
    let m10 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let m20 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    let m30 = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
    let m01 = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : 0;
    let m11 = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 1;
    let m21 = arguments.length > 6 && arguments[6] !== undefined ? arguments[6] : 0;
    let m31 = arguments.length > 7 && arguments[7] !== undefined ? arguments[7] : 0;
    let m02 = arguments.length > 8 && arguments[8] !== undefined ? arguments[8] : 0;
    let m12 = arguments.length > 9 && arguments[9] !== undefined ? arguments[9] : 0;
    let m22 = arguments.length > 10 && arguments[10] !== undefined ? arguments[10] : 1;
    let m32 = arguments.length > 11 && arguments[11] !== undefined ? arguments[11] : 0;
    let m03 = arguments.length > 12 && arguments[12] !== undefined ? arguments[12] : 0;
    let m13 = arguments.length > 13 && arguments[13] !== undefined ? arguments[13] : 0;
    let m23 = arguments.length > 14 && arguments[14] !== undefined ? arguments[14] : 0;
    let m33 = arguments.length > 15 && arguments[15] !== undefined ? arguments[15] : 1;
    this[0] = m00;
    this[1] = m10;
    this[2] = m20;
    this[3] = m30;
    this[4] = m01;
    this[5] = m11;
    this[6] = m21;
    this[7] = m31;
    this[8] = m02;
    this[9] = m12;
    this[10] = m22;
    this[11] = m32;
    this[12] = m03;
    this[13] = m13;
    this[14] = m23;
    this[15] = m33;
    return this.check();
  }

  copy(array) {
    return this.setColumnMajor(...array);
  }

  set() {
    return this.setColumnMajor(...arguments);
  }

  getElement(i, j) {
    let columnMajor = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
    return columnMajor ? this[i][j] : this[j][i];
  }

  setElement(i, j, value) {
    let columnMajor = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

    if (columnMajor) {
      this[i][j] = checkNumber(value);
    } else {
      this[j][i] = checkNumber(value);
    }

    return this;
  }

  determinant() {
    return mat4.determinant(this);
  }

  identity() {
    return this.copy(IDENTITY);
  }

  fromQuaternion(q) {
    mat4.fromQuat(this, q);
    return this.check();
  }

  frustum(_ref) {
    let left = _ref.left,
        right = _ref.right,
        bottom = _ref.bottom,
        top = _ref.top,
        near = _ref.near,
        far = _ref.far;
    mat4.frustum(this, left, right, bottom, top, near, far);
    return this.check();
  }

  lookAt(_ref2) {
    let eye = _ref2.eye,
        _ref2$center = _ref2.center,
        center = _ref2$center === void 0 ? [0, 0, 0] : _ref2$center,
        _ref2$up = _ref2.up,
        up = _ref2$up === void 0 ? [0, 1, 0] : _ref2$up;
    mat4.lookAt(this, eye, center, up);
    return this.check();
  }

  ortho(_ref3) {
    let left = _ref3.left,
        right = _ref3.right,
        bottom = _ref3.bottom,
        top = _ref3.top,
        _ref3$near = _ref3.near,
        near = _ref3$near === void 0 ? 0.1 : _ref3$near,
        _ref3$far = _ref3.far,
        far = _ref3$far === void 0 ? 500 : _ref3$far;
    mat4.ortho(this, left, right, bottom, top, near, far);
    return this.check();
  }

  orthographic(_ref4) {
    let _ref4$fovy = _ref4.fovy,
        fovy = _ref4$fovy === void 0 ? 45 * Math.PI / 180 : _ref4$fovy,
        _ref4$aspect = _ref4.aspect,
        aspect = _ref4$aspect === void 0 ? 1 : _ref4$aspect,
        _ref4$focalDistance = _ref4.focalDistance,
        focalDistance = _ref4$focalDistance === void 0 ? 1 : _ref4$focalDistance,
        _ref4$near = _ref4.near,
        near = _ref4$near === void 0 ? 0.1 : _ref4$near,
        _ref4$far = _ref4.far,
        far = _ref4$far === void 0 ? 500 : _ref4$far;

    if (fovy > Math.PI * 2) {
      throw Error('radians');
    }

    const halfY = fovy / 2;
    const top = focalDistance * Math.tan(halfY);
    const right = top * aspect;
    return new Matrix4().ortho({
      left: -right,
      right,
      bottom: -top,
      top,
      near,
      far
    });
  }

  perspective() {
    let _ref5 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
        fovy = _ref5.fovy,
        _ref5$fov = _ref5.fov,
        fov = _ref5$fov === void 0 ? 45 * Math.PI / 180 : _ref5$fov,
        _ref5$aspect = _ref5.aspect,
        aspect = _ref5$aspect === void 0 ? 1 : _ref5$aspect,
        _ref5$near = _ref5.near,
        near = _ref5$near === void 0 ? 0.1 : _ref5$near,
        _ref5$far = _ref5.far,
        far = _ref5$far === void 0 ? 500 : _ref5$far;

    fovy = fovy || fov;

    if (fovy > Math.PI * 2) {
      throw Error('radians');
    }

    mat4.perspective(this, fovy, aspect, near, far);
    return this.check();
  }

  transpose() {
    mat4.transpose(this, this);
    return this.check();
  }

  invert() {
    mat4.invert(this, this);
    return this.check();
  }

  multiplyLeft(a) {
    mat4.multiply(this, a, this);
    return this.check();
  }

  multiplyRight(a) {
    mat4.multiply(this, this, a);
    return this.check();
  }

  rotateX(radians) {
    mat4.rotateX(this, this, radians);
    return this.check();
  }

  rotateY(radians) {
    mat4.rotateY(this, this, radians);
    return this.check();
  }

  rotateZ(radians) {
    mat4.rotateZ(this, this, radians);
    return this.check();
  }

  rotateXYZ(_ref6) {
    let _ref7 = _slicedToArray(_ref6, 3),
        rx = _ref7[0],
        ry = _ref7[1],
        rz = _ref7[2];

    return this.rotateX(rx).rotateY(ry).rotateZ(rz);
  }

  rotateAxis(radians, axis) {
    mat4.rotate(this, this, radians, axis);
    return this.check();
  }

  scale(factor) {
    if (Array.isArray(factor)) {
      mat4.scale(this, this, factor);
    } else {
      mat4.scale(this, this, [factor, factor, factor]);
    }

    return this.check();
  }

  translate(vec) {
    mat4.translate(this, this, vec);
    return this.check();
  }

  transformVector2(vector, out) {
    out = out || new Vector2();
    vec2.transformMat4(out, vector, this);
    validateVector(out, 2);
    return out;
  }

  transformVector3(vector, out) {
    out = out || new Vector3();
    vec3.transformMat4(out, vector, this);
    validateVector(out, 3);
    return out;
  }

  transformVector4(vector, out) {
    out = out || new Vector4();
    vec4.transformMat4(out, vector, this);
    validateVector(out, 4);
    return out.check();
  }

  transformVector(vector, out) {
    switch (vector.length) {
      case 2:
        return this.transformVector2(vector, out);

      case 3:
        return this.transformVector3(vector, out);

      case 4:
        return this.transformVector4(vector, out);

      default:
        throw new Error('Illegal vector');
    }
  }

  transformDirection(vector, out) {
    return this._transformVector(vector, out, 0);
  }

  transformPoint(vector, out) {
    return this._transformVector(vector, out, 1);
  }

  _transformVector(vector, out, w) {
    switch (vector.length) {
      case 2:
        out = out || new Vector2();
        vec4.transformMat4(out, [vector[0], vector[1], 0, w], this);
        out.length = 2;
        validateVector(out, 2);
        break;

      case 3:
        out = out || new Vector3();
        vec4.transformMat4(out, [vector[0], vector[1], vector[2], w], this);
        out.length = 3;
        validateVector(out, 3);
        break;

      case 4:
        if (Boolean(w) !== Boolean(vector[3])) {
          throw new Error('math.gl: Matrix4.transformPoint - invalid vector');
        }

        out = out || new Vector4();
        vec4.transformMat4(out, vector, this);
        validateVector(out, 4);
        break;

      default:
        throw new Error('Illegal vector');
    }

    return out;
  }

  makeRotationX(radians) {
    return this.identity().rotateX(radians);
  }

  makeTranslation(x, y, z) {
    return this.identity().translate([x, y, z]);
  }

}
//# sourceMappingURL=matrix4.js.map