import _slicedToArray from "@babel/runtime/helpers/esm/slicedToArray";
import { formatValue, equals, config } from './lib/common';
import { degrees, radians, clamp } from './lib/common';
import Vector3 from './vector3';
import * as vec3 from 'gl-matrix/vec3';
const EPSILON = 0.000001;
const EARTH_RADIUS_METERS = 6.371e6;
export default class SphericalCoordinates {
  constructor() {
    let _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
        phi = _ref.phi,
        theta = _ref.theta,
        radius = _ref.radius,
        bearing = _ref.bearing,
        pitch = _ref.pitch,
        altitude = _ref.altitude,
        _ref$radiusScale = _ref.radiusScale,
        radiusScale = _ref$radiusScale === void 0 ? EARTH_RADIUS_METERS : _ref$radiusScale;

    if (arguments.length === 0) {
      this.phi = 0;
      this.theta = 0;
      this.radius = 1;
    } else if (Number.isFinite(phi) || Number.isFinite(theta)) {
      this.phi = phi || 0;
      this.theta = theta || 0;
    } else if (Number.isFinite(bearing) || Number.isFinite(pitch)) {
      this.bearing = bearing || 0;
      this.pitch = pitch || 0;
    }

    this.radius = radius || 1;
    this.radiusScale = radiusScale || 1;
    this.check();
  }

  toString() {
    return this.formatString(config);
  }

  formatString(_ref2) {
    let printTypes = _ref2.printTypes,
        printDegrees = _ref2.printDegrees;
    const f = formatValue;
    return `${printTypes ? 'Spherical' : ''}\
[rho:${f(this.radius)},theta:${f(this.theta)},phi:${f(this.phi)}]`;
  }

  equals(other) {
    return equals(this.radius, other.radius) && equals(this.theta, other.theta) && equals(this.phi, other.phi);
  }

  exactEquals(other) {
    return this.radius === other.radius && this.theta === other.theta && this.phi === other.phi;
  }

  get bearing() {
    return 180 - degrees(this.phi);
  }

  set bearing(v) {
    this.phi = Math.PI - radians(v);
  }

  get pitch() {
    return degrees(this.theta);
  }

  set pitch(v) {
    this.theta = radians(v);
  }

  get longitude() {
    return degrees(this.phi);
  }

  get latitude() {
    return degrees(this.theta);
  }

  get lng() {
    return degrees(this.phi);
  }

  get lat() {
    return degrees(this.theta);
  }

  get z() {
    return (this.radius - 1) * this.radiusScale;
  }

  set(radius, phi, theta) {
    this.radius = radius;
    this.phi = phi;
    this.theta = theta;
    return this.check();
  }

  clone() {
    return new this.constructor().copy(this);
  }

  copy(other) {
    this.radius = other.radius;
    this.phi = other.phi;
    this.theta = other.theta;
    return this.check();
  }

  fromLngLatZ(_ref3) {
    let _ref4 = _slicedToArray(_ref3, 3),
        lng = _ref4[0],
        lat = _ref4[1],
        z = _ref4[2];

    this.radius = 1 + z / this.radiusScale;
    this.phi = radians(lat);
    this.theta = radians(lng);
  }

  fromVector3(v) {
    this.radius = vec3.length(v);

    if (this.radius === 0) {
      this.theta = 0;
      this.phi = 0;
    } else {
      this.theta = Math.atan2(v[0], v[1]);
      this.phi = Math.acos(clamp(v[2] / this.radius, -1, 1));
    }

    return this.check();
  }

  toVector3() {
    return new Vector3(0, 0, this.radius).rotateX({
      radians: this.theta
    }).rotateZ({
      radians: this.phi
    });
  }

  makeSafe() {
    this.phi = Math.max(EPSILON, Math.min(Math.PI - EPSILON, this.phi));
  }

  check() {
    if (!Number.isFinite(this.phi) || !Number.isFinite(this.theta) || !(this.radius > 0)) {
      throw new Error('SphericalCoordinates: some fields set to invalid numbers');
    }

    return this;
  }

}
//# sourceMappingURL=spherical-coordinates.js.map