"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configure = configure;
exports.checkNumber = checkNumber;
exports.formatValue = formatValue;
exports.formatAngle = formatAngle;
exports.isArray = isArray;
exports.clone = clone;
exports.radians = radians;
exports.degrees = degrees;
exports.sin = sin;
exports.cos = cos;
exports.tan = tan;
exports.asin = asin;
exports.acos = acos;
exports.atan = atan;
exports.clamp = clamp;
exports.lerp = lerp;
exports.equals = equals;
exports.config = void 0;
var config = {};
exports.config = config;
config.EPSILON = 1e-12;
config.debug = true;
config.precision = 4;
config.printTypes = false;
config.printDegrees = false;
config.printRowMajor = true;

function configure(options) {
  if ('epsilon' in options) {
    config.EPSILON = options.epsilon;
  }

  if ('debug' in options) {
    config.debug = options.debug;
  }
}

function checkNumber(value) {
  if (!Number.isFinite(value)) {
    throw new Error("Invalid number ".concat(value));
  }

  return value;
}

function round(value) {
  return Math.round(value / config.EPSILON) * config.EPSILON;
}

function formatValue(value) {
  var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref$precision = _ref.precision,
      precision = _ref$precision === void 0 ? config.precision || 4 : _ref$precision;

  value = round(value);
  return parseFloat(value.toPrecision(precision));
}

function formatAngle(value) {
  var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref2$precision = _ref2.precision,
      precision = _ref2$precision === void 0 ? config.precision || 4 : _ref2$precision,
      _ref2$printDegrees = _ref2.printDegrees,
      printDegrees = _ref2$printDegrees === void 0 ? config.printAngles : _ref2$printDegrees;

  value = printDegrees ? degrees(value) : value;
  value = round(value);
  return "".concat(parseFloat(value.toPrecision(precision))).concat(printDegrees ? '°' : '');
}

function isArray(value) {
  return Array.isArray(value) || ArrayBuffer.isView(value) && value.length !== undefined;
}

function clone(array) {
  return array.clone ? array.clone() : new Array(array);
}

function map(value, func) {
  if (isArray(value)) {
    var result = clone(value);

    for (var i = 0; i < result.length; ++i) {
      result[i] = func(result[i], i, result);
    }

    return result;
  }

  return func(value);
}

function radians(degrees) {
  return map(degrees, function (degrees) {
    return degrees / 180 * Math.PI;
  });
}

function degrees(radians) {
  return map(radians, function (radians) {
    return radians * 180 / Math.PI;
  });
}

function sin(radians) {
  return map(radians, function (angle) {
    return Math.sin(angle);
  });
}

function cos(radians) {
  return map(radians, function (angle) {
    return Math.cos(angle);
  });
}

function tan(radians) {
  return map(radians, function (angle) {
    return Math.tan(angle);
  });
}

function asin(radians) {
  return map(radians, function (angle) {
    return Math.asin(angle);
  });
}

function acos(radians) {
  return map(radians, function (angle) {
    return Math.acos(angle);
  });
}

function atan(radians) {
  return map(radians, function (angle) {
    return Math.atan(angle);
  });
}

function clamp(value, min, max) {
  return map(value, function (value) {
    return Math.max(min, Math.min(max, value));
  });
}

function lerp(a, b, t) {
  if (isArray(a)) {
    return a.map(function (ai, i) {
      return lerp(ai, b[i], t);
    });
  }

  return t * b + (1 - t) * a;
}

function equals(a, b) {
  if (isArray(a) && isArray(b)) {
    if (a === b) {
      return true;
    }

    if (a.length !== b.length) {
      return false;
    }

    for (var i = 0; i < a.length; ++i) {
      if (!equals(a[i], b[i])) {
        return false;
      }
    }

    return true;
  }

  return Math.abs(a - b) <= config.EPSILON * Math.max(1.0, Math.abs(a), Math.abs(b));
}
//# sourceMappingURL=common.js.map