const config = {};
config.EPSILON = 1e-12;
config.debug = true;
config.precision = 4;
config.printTypes = false;
config.printDegrees = false;
config.printRowMajor = true;
export { config };
export function configure(options) {
  if ('epsilon' in options) {
    config.EPSILON = options.epsilon;
  }

  if ('debug' in options) {
    config.debug = options.debug;
  }
}
export function checkNumber(value) {
  if (!Number.isFinite(value)) {
    throw new Error(`Invalid number ${value}`);
  }

  return value;
}

function round(value) {
  return Math.round(value / config.EPSILON) * config.EPSILON;
}

export function formatValue(value) {
  let _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref$precision = _ref.precision,
      precision = _ref$precision === void 0 ? config.precision || 4 : _ref$precision;

  value = round(value);
  return parseFloat(value.toPrecision(precision));
}
export function formatAngle(value) {
  let _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _ref2$precision = _ref2.precision,
      precision = _ref2$precision === void 0 ? config.precision || 4 : _ref2$precision,
      _ref2$printDegrees = _ref2.printDegrees,
      printDegrees = _ref2$printDegrees === void 0 ? config.printAngles : _ref2$printDegrees;

  value = printDegrees ? degrees(value) : value;
  value = round(value);
  return `${parseFloat(value.toPrecision(precision))}${printDegrees ? '°' : ''}`;
}
export function isArray(value) {
  return Array.isArray(value) || ArrayBuffer.isView(value) && value.length !== undefined;
}
export function clone(array) {
  return array.clone ? array.clone() : new Array(array);
}

function map(value, func) {
  if (isArray(value)) {
    const result = clone(value);

    for (let i = 0; i < result.length; ++i) {
      result[i] = func(result[i], i, result);
    }

    return result;
  }

  return func(value);
}

export function radians(degrees) {
  return map(degrees, degrees => degrees / 180 * Math.PI);
}
export function degrees(radians) {
  return map(radians, radians => radians * 180 / Math.PI);
}
export function sin(radians) {
  return map(radians, angle => Math.sin(angle));
}
export function cos(radians) {
  return map(radians, angle => Math.cos(angle));
}
export function tan(radians) {
  return map(radians, angle => Math.tan(angle));
}
export function asin(radians) {
  return map(radians, angle => Math.asin(angle));
}
export function acos(radians) {
  return map(radians, angle => Math.acos(angle));
}
export function atan(radians) {
  return map(radians, angle => Math.atan(angle));
}
export function clamp(value, min, max) {
  return map(value, value => Math.max(min, Math.min(max, value)));
}
export function lerp(a, b, t) {
  if (isArray(a)) {
    return a.map((ai, i) => lerp(ai, b[i], t));
  }

  return t * b + (1 - t) * a;
}
export function equals(a, b) {
  if (isArray(a) && isArray(b)) {
    if (a === b) {
      return true;
    }

    if (a.length !== b.length) {
      return false;
    }

    for (let i = 0; i < a.length; ++i) {
      if (!equals(a[i], b[i])) {
        return false;
      }
    }

    return true;
  }

  return Math.abs(a - b) <= config.EPSILON * Math.max(1.0, Math.abs(a), Math.abs(b));
}
//# sourceMappingURL=common.js.map