import MathArray from './lib/math-array';
import { checkNumber } from './lib/common';
import * as quat from 'gl-matrix/quat';
const IDENTITY_QUATERNION = [0, 0, 0, 1];
export function validateQuaternion(q) {
  return q.length === 4 && Number.isFinite(q[0]) && Number.isFinite(q[1]) && Number.isFinite(q[2]) && Number.isFinite(q[3]);
}
export default class Quaternion extends MathArray {
  constructor() {
    let x = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;
    let y = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let z = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    let w = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 1;
    super();

    if (Array.isArray(x) && arguments.length === 1) {
      this.copy(x);
    } else {
      this.set(x, y, z, w);
    }
  }

  fromMatrix3(m) {
    quat.fromMat3(this, m);
    return this.check();
  }

  fromValues(x, y, z, w) {
    return this.set(x, y, z, w);
  }

  identity() {
    quat.identity(this);
    return this.check();
  }

  get ELEMENTS() {
    return 4;
  }

  get x() {
    return this[0];
  }

  set x(value) {
    return this[0] = checkNumber(value);
  }

  get y() {
    return this[1];
  }

  set y(value) {
    return this[1] = checkNumber(value);
  }

  get z() {
    return this[2];
  }

  set z(value) {
    return this[2] = checkNumber(value);
  }

  get w() {
    return this[3];
  }

  set w(value) {
    return this[3] = checkNumber(value);
  }

  length() {
    return quat.length(this);
  }

  squaredLength(a) {
    return quat.squaredLength(this);
  }

  dot(a, b) {
    if (b !== undefined) {
      throw new Error('Quaternion.dot only takes one argument');
    }

    return quat.dot(this, a);
  }

  rotationTo(vectorA, vectorB) {
    quat.rotationTo(this, vectorA, vectorB);
    return this.check();
  }

  add(a, b) {
    if (b !== undefined) {
      throw new Error('Quaternion.add only takes one argument');
    }

    quat.add(this, a);
    return this.check();
  }

  calculateW() {
    quat.calculateW(this, this);
    return this.check();
  }

  conjugate() {
    quat.conjugate(this, this);
    return this.check();
  }

  invert() {
    quat.invert(this, this);
    return this.check();
  }

  lerp(a, b, t) {
    quat.lerp(this, a, b, t);
    return this.check();
  }

  multiply(a, b) {
    if (b !== undefined) {
      throw new Error('Quaternion.multiply only takes one argument');
    }

    quat.multiply(this, this, a);
    return this.check();
  }

  normalize() {
    quat.normalize(this, this);
    return this.check();
  }

  rotateX(rad) {
    quat.rotateX(this, this, rad);
    return this.check();
  }

  rotateY(rad) {
    quat.rotateY(this, this, rad);
    return this.check();
  }

  rotateZ(rad) {
    quat.rotateZ(this, this, rad);
    return this.check();
  }

  scale(b) {
    quat.scale(this, this, b);
    return this.check();
  }

  set(i, j, k, l) {
    quat.set(this, i, j, k, l);
    return this.check();
  }

  setAxisAngle(axis, rad) {
    quat.setAxisAngle(this, axis, rad);
    return this.check();
  }

  slerp(_ref) {
    let _ref$start = _ref.start,
        start = _ref$start === void 0 ? IDENTITY_QUATERNION : _ref$start,
        target = _ref.target,
        ratio = _ref.ratio;
    quat.slerp(this, start, target, ratio);
    return this.check();
  }

}
//# sourceMappingURL=quaternion.js.map