var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

// Copyright (c) 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

import { Manager } from './utils/hammer';

import WheelInput from './inputs/wheel-input';
import MoveInput from './inputs/move-input';
import KeyInput from './inputs/key-input';
import ContextmenuInput from './inputs/contextmenu-input';

import { BASIC_EVENT_ALIASES, EVENT_RECOGNIZER_MAP, GESTURE_EVENT_ALIASES, RECOGNIZERS, RECOGNIZER_COMPATIBLE_MAP, RECOGNIZER_FALLBACK_MAP } from './constants';

import { whichButtons, getOffsetPosition } from './utils/event-utils';

var DEFAULT_OPTIONS = {
  // event handlers
  events: null,
  // custom recognizers
  recognizers: null,
  // Manager class
  Manager: Manager,
  // recognize right button gestures
  rightButton: false,
  // allow browser default touch action
  // https://github.com/uber/react-map-gl/issues/506
  touchAction: 'none',
  // block scrolling - this is a legacy behavior and will be removed in the next version
  legacyBlockScroll: true
};

// Unified API for subscribing to events about both
// basic input events (e.g. 'mousemove', 'touchstart', 'wheel')
// and gestural input (e.g. 'click', 'tap', 'panstart').
// Delegates gesture related event registration and handling to Hammer.js.

var EventManager = function () {
  function EventManager() {
    var element = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    _classCallCheck(this, EventManager);

    this.options = Object.assign({}, DEFAULT_OPTIONS, options);
    this.eventHandlers = [];

    this._onBasicInput = this._onBasicInput.bind(this);
    this._onOtherEvent = this._onOtherEvent.bind(this);

    this.setElement(element);

    // Register all passed events.
    var events = options.events;

    if (events) {
      this.on(events);
    }
  }

  _createClass(EventManager, [{
    key: 'setElement',
    value: function setElement(element) {
      var _this = this;

      if (this.element) {
        // unregister all events
        this.destroy();
      }
      this.element = element;
      if (!element) {
        return;
      }

      var options = this.options;

      var ManagerClass = options.Manager;

      this.manager = new ManagerClass(element, {
        touchAction: options.touchAction,
        recognizers: options.recognizers || RECOGNIZERS
      }).on('hammer.input', this._onBasicInput);

      if (!options.recognizers) {
        // Set default recognize withs
        // http://hammerjs.github.io/recognize-with/
        Object.keys(RECOGNIZER_COMPATIBLE_MAP).forEach(function (name) {
          var recognizer = _this.manager.get(name);
          if (recognizer) {
            RECOGNIZER_COMPATIBLE_MAP[name].forEach(function (otherName) {
              recognizer.recognizeWith(otherName);
            });
          }
        });
      }

      // Handle events not handled by Hammer.js:
      // - mouse wheel
      // - pointer/touch/mouse move
      this.wheelInput = new WheelInput(element, this._onOtherEvent, {
        enable: false,
        legacyBlockScroll: options.legacyBlockScroll
      });
      this.moveInput = new MoveInput(element, this._onOtherEvent, { enable: false });
      this.keyInput = new KeyInput(element, this._onOtherEvent, { enable: false });
      this.contextmenuInput = new ContextmenuInput(element, this._onOtherEvent, {
        rightButton: options.rightButton
      });

      // Register all existing events
      this.eventHandlers.forEach(function (_ref) {
        var recognizerName = _ref.recognizerName,
            eventAlias = _ref.eventAlias,
            wrappedHandler = _ref.wrappedHandler;

        // Enable recognizer for this event.
        _this._toggleRecognizer(recognizerName, true);
        _this.manager.on(eventAlias, wrappedHandler);
      });
    }

    // Tear down internal event management implementations.

  }, {
    key: 'destroy',
    value: function destroy() {
      if (this.element) {
        // wheelInput etc. are created in setElement() and therefore
        // cannot exist if there is no element
        this.wheelInput.destroy();
        this.moveInput.destroy();
        this.keyInput.destroy();
        this.contextmenuInput.destroy();
        this.manager.destroy();

        this.wheelInput = null;
        this.moveInput = null;
        this.keyInput = null;
        this.contextmenuInput = null;
        this.manager = null;
        this.element = null;
      }
    }

    // Register an event handler function to be called on `event`.

  }, {
    key: 'on',
    value: function on(event, handler, srcElement) {
      if (typeof event === 'string') {
        this._addEventHandler(event, handler, srcElement);
      } else {
        srcElement = handler;
        // If `event` is a map, call `on()` for each entry.
        for (var eventName in event) {
          this._addEventHandler(eventName, event[eventName], srcElement);
        }
      }
    }

    /**
     * Deregister a previously-registered event handler.
     * @param {string|Object} event   An event name (String) or map of event names to handlers
     * @param {Function} [handler]    The function to be called on `event`.
     */

  }, {
    key: 'off',
    value: function off(event, handler) {
      if (typeof event === 'string') {
        this._removeEventHandler(event, handler);
      } else {
        // If `event` is a map, call `off()` for each entry.
        for (var eventName in event) {
          this._removeEventHandler(eventName, event[eventName]);
        }
      }
    }

    /*
     * Enable/disable recognizer for the given event
     */

  }, {
    key: '_toggleRecognizer',
    value: function _toggleRecognizer(name, enabled) {
      var manager = this.manager;

      if (!manager) {
        return;
      }
      var recognizer = manager.get(name);
      if (recognizer) {
        recognizer.set({ enable: enabled });

        var fallbackRecognizers = RECOGNIZER_FALLBACK_MAP[name];
        if (fallbackRecognizers && !this.options.recognizers) {
          // Set default require failures
          // http://hammerjs.github.io/require-failure/
          fallbackRecognizers.forEach(function (otherName) {
            var otherRecognizer = manager.get(otherName);
            if (enabled) {
              // Wait for this recognizer to fail
              otherRecognizer.requireFailure(name);
            } else {
              // Do not wait for this recognizer to fail
              otherRecognizer.dropRequireFailure(name);
            }
          });
        }
      }
      this.wheelInput.enableEventType(name, enabled);
      this.moveInput.enableEventType(name, enabled);
      this.keyInput.enableEventType(name, enabled);
    }

    /**
     * Process the event registration for a single event + handler.
     */

  }, {
    key: '_addEventHandler',
    value: function _addEventHandler(event, handler) {
      var srcElement = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;
      var manager = this.manager,
          eventHandlers = this.eventHandlers;

      var wrappedHandler = this._wrapEventHandler(event, handler, srcElement);
      // Alias to a recognized gesture as necessary.
      var eventAlias = GESTURE_EVENT_ALIASES[event] || event;
      // Get recognizer for this event
      var recognizerName = EVENT_RECOGNIZER_MAP[eventAlias] || eventAlias;
      // Enable recognizer for this event.
      this._toggleRecognizer(recognizerName, true);

      // Find ancestors
      var ancestorEventHandlers = eventHandlers.filter(function (entry) {
        return entry.eventAlias === eventAlias && entry.srcElement !== srcElement && (!entry.srcElement || entry.srcElement.contains(srcElement));
      });

      // Save wrapped handler
      eventHandlers.push({ event: event, eventAlias: eventAlias, recognizerName: recognizerName, srcElement: srcElement,
        handler: handler, wrappedHandler: wrappedHandler });

      // Sort handlers by DOM hierarchy
      // So the event will always fire first on child nodes
      ancestorEventHandlers.forEach(function (entry) {
        return manager.off(eventAlias, entry.wrappedHandler);
      });
      if (manager) {
        manager.on(eventAlias, wrappedHandler);
      }
      ancestorEventHandlers.forEach(function (entry) {
        return manager.on(eventAlias, entry.wrappedHandler);
      });
    }

    /**
     * Process the event deregistration for a single event + handler.
     */

  }, {
    key: '_removeEventHandler',
    value: function _removeEventHandler(event, handler) {
      var manager = this.manager,
          eventHandlers = this.eventHandlers;

      var eventHandlerRemoved = false;

      // Find saved handler if any.
      for (var i = eventHandlers.length; i--;) {
        var entry = eventHandlers[i];
        if (entry.event === event && entry.handler === handler) {
          // Deregister event handler.
          if (manager) {
            manager.off(entry.eventAlias, entry.wrappedHandler);
          }
          // Delete saved handler
          eventHandlers.splice(i, 1);
          eventHandlerRemoved = true;
        }
      }

      if (eventHandlerRemoved) {
        // Alias to a recognized gesture as necessary.
        var eventAlias = GESTURE_EVENT_ALIASES[event] || event;
        // Get recognizer for this event
        var recognizerName = EVENT_RECOGNIZER_MAP[eventAlias] || eventAlias;
        // Disable recognizer if no more handlers are attached to its events
        var isRecognizerUsed = eventHandlers.find(function (entry) {
          return entry.recognizerName === recognizerName;
        });
        if (!isRecognizerUsed) {
          this._toggleRecognizer(recognizerName, false);
        }
      }
    }

    /**
     * Returns an event handler that aliases events and add props before passing
     * to the real handler.
     */

  }, {
    key: '_wrapEventHandler',
    value: function _wrapEventHandler(type, handler, srcElement) {
      var _this2 = this;

      return function (event) {
        var mjolnirEvent = event.mjolnirEvent;


        if (!mjolnirEvent) {
          mjolnirEvent = _this2._normalizeEvent(event);
          event.mjolnirEvent = mjolnirEvent;
        }

        var isStopped = mjolnirEvent.handled && mjolnirEvent.handled !== srcElement;

        if (!isStopped) {
          var isFromDecendant = !srcElement || srcElement.contains(event.srcEvent.target);
          if (isFromDecendant) {
            handler(Object.assign({}, mjolnirEvent, {
              type: type,
              stopPropagation: function stopPropagation() {
                if (!mjolnirEvent.handled) {
                  mjolnirEvent.handled = srcElement;
                }
              }
            }));
          }
        }
      };
    }

    /**
     * Normalizes hammerjs and custom events to have predictable fields.
     */

  }, {
    key: '_normalizeEvent',
    value: function _normalizeEvent(event) {
      var element = this.element;


      return Object.assign({}, event, whichButtons(event), getOffsetPosition(event, element), {
        handled: false,
        rootElement: element
      });
    }

    /**
     * Handle basic events using the 'hammer.input' Hammer.js API:
     * Before running Recognizers, Hammer emits a 'hammer.input' event
     * with the basic event info. This function emits all basic events
     * aliased to the "class" of event received.
     * See constants.BASIC_EVENT_CLASSES basic event class definitions.
     */

  }, {
    key: '_onBasicInput',
    value: function _onBasicInput(event) {
      var srcEvent = event.srcEvent;

      var alias = BASIC_EVENT_ALIASES[srcEvent.type];
      if (alias) {
        // fire all events aliased to srcEvent.type
        this.manager.emit(alias, event);
      }
    }

    /**
     * Handle events not supported by Hammer.js,
     * and pipe back out through same (Hammer) channel used by other events.
     */

  }, {
    key: '_onOtherEvent',
    value: function _onOtherEvent(event) {
      this.manager.emit(event.type, event);
    }
  }]);

  return EventManager;
}();

export default EventManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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