/* Constants */
var DOWN_EVENT = 1;
var MOVE_EVENT = 2;
var UP_EVENT = 4;
var MOUSE_EVENTS = {
  pointerdown: DOWN_EVENT,
  pointermove: MOVE_EVENT,
  pointerup: UP_EVENT,
  mousedown: DOWN_EVENT,
  mousemove: MOVE_EVENT,
  mouseup: UP_EVENT
};

// MouseEvent.which https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/which
var MOUSE_EVENT_WHICH_LEFT = 1;
var MOUSE_EVENT_WHICH_MIDDLE = 2;
var MOUSE_EVENT_WHICH_RIGHT = 3;
// MouseEvent.button https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/button
var MOUSE_EVENT_BUTTON_LEFT = 0;
var MOUSE_EVENT_BUTTON_MIDDLE = 1;
var MOUSE_EVENT_BUTTON_RIGHT = 2;
// MouseEvent.buttons https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/buttons
var MOUSE_EVENT_BUTTONS_LEFT_MASK = 1;
var MOUSE_EVENT_BUTTONS_RIGHT_MASK = 2;
var MOUSE_EVENT_BUTTONS_MIDDLE_MASK = 4;

/**
 * Extract the involved mouse button
 */
export function whichButtons(event) {
  var eventType = MOUSE_EVENTS[event.srcEvent.type];
  if (!eventType) {
    // Not a mouse evet
    return null;
  }

  var _event$srcEvent = event.srcEvent,
      buttons = _event$srcEvent.buttons,
      button = _event$srcEvent.button,
      which = _event$srcEvent.which;

  var leftButton = false;
  var middleButton = false;
  var rightButton = false;

  if (
  // button is up, need to find out which one was pressed before
  eventType === UP_EVENT ||
  // moving but does not support `buttons` API
  eventType === MOVE_EVENT && !Number.isFinite(buttons)) {
    leftButton = which === MOUSE_EVENT_WHICH_LEFT;
    middleButton = which === MOUSE_EVENT_WHICH_MIDDLE;
    rightButton = which === MOUSE_EVENT_WHICH_RIGHT;
  } else if (eventType === MOVE_EVENT) {
    leftButton = Boolean(buttons & MOUSE_EVENT_BUTTONS_LEFT_MASK);
    middleButton = Boolean(buttons & MOUSE_EVENT_BUTTONS_MIDDLE_MASK);
    rightButton = Boolean(buttons & MOUSE_EVENT_BUTTONS_RIGHT_MASK);
  } else if (eventType === DOWN_EVENT) {
    leftButton = button === MOUSE_EVENT_BUTTON_LEFT;
    middleButton = button === MOUSE_EVENT_BUTTON_MIDDLE;
    rightButton = button === MOUSE_EVENT_BUTTON_RIGHT;
  }

  return { leftButton: leftButton, middleButton: middleButton, rightButton: rightButton };
}

/**
 * Calculate event position relative to the root element
 */
export function getOffsetPosition(event, rootElement) {
  var srcEvent = event.srcEvent;

  // `center` is a hammer.js event property

  if (!event.center && !Number.isFinite(srcEvent.clientX)) {
    // Not a gestural event
    return null;
  }

  var center = event.center || {
    x: srcEvent.clientX,
    y: srcEvent.clientY
  };

  var rect = rootElement.getBoundingClientRect();

  // Fix scale for map affected by a CSS transform.
  // See https://stackoverflow.com/a/26893663/3528533
  var scaleX = rect.width / rootElement.offsetWidth;
  var scaleY = rect.height / rootElement.offsetHeight;

  // Calculate center relative to the root element
  var offsetCenter = {
    x: (center.x - rect.left - rootElement.clientLeft) / scaleX,
    y: (center.y - rect.top - rootElement.clientTop) / scaleY
  };

  return { center: center, offsetCenter: offsetCenter };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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