/*
scrypt_common.cc

Copyright (C) 2013 Barry Steyn (http://doctrina.org/Scrypt-Authentication-For-Node.html)

This source code is provided 'as-is', without any express or implied
warranty. In no event will the author be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this source code must not be misrepresented; you must not
   claim that you wrote the original source code. If you use this source code
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original source code.
3. This notice may not be removed or altered from any source distribution.

Barry Steyn barry.steyn@gmail.com
*/


#include <iostream>

#include <nan.h>

extern "C" {
  #include <errno.h>
}

#include <string>
#include <string.h>

#include <stdlib.h>
#include <stdio.h>

#include <iostream>

//
// Anonymous namespace
//
namespace {
  //
  // Returns error descriptions as generated by Scrypt
  //
  const char* InternalErrorDescr(const unsigned int error) {
    switch(error) {
      case 0:
        return "success";
      case 1:
        return "getrlimit or sysctl(hw.usermem) failed";
      case 2:
        return "clock_getres or clock_gettime failed";
      case 3:
        return "error computing derived key";
      case 4:
        return "could not read salt from /dev/urandom";
      case 5:
        return "error in OpenSSL";
      case 6:
        return "malloc failed";
      case 7:
        return "data is not a valid scrypt-encrypted block";
      case 8:
        return "unrecognized scrypt format";
      case 9:
        return "decrypting file would take too much memory";
      case 10:
        return "decrypting file would take too long";
      case 11:
        return "password is incorrect";
      case 12:
        return "error writing output file";
      case 13:
        return "error reading input file";
      default:
        return "error unkown";
    }
  }

  //
  // Returns error descriptions as generated by Scrypt
  //
  const char* ScryptErrorDescr(const unsigned int error) {

    unsigned int mask = -1,
                 base_error = (mask >> 16) & error,
                 sub_error = (((mask << 16) & error) >> 16);
    std::string scrypt_err_description = InternalErrorDescr(base_error);

    if (sub_error) {
      scrypt_err_description += " - " + std::string(strerror(sub_error));
    }

    return scrypt_err_description.c_str();
  }
} /* end anonymous namespace */

namespace NodeScrypt {

  //
  // Creates a Scrypt specific JavaScript Error object
  //
  v8::Local<v8::Value> ScryptError(const unsigned int error) {
    Nan::EscapableHandleScope scope;
    v8::Local<v8::Value> scryptError = Nan::Error(ScryptErrorDescr(error));
    return scope.Escape(scryptError);
  }
} //end Scrypt namespace
