'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

// Copyright (c) 2017 Uber Technologies, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

var isBrowser = typeof window !== 'undefined' && window.addEventListener;

var timers = new Map();

/**
 * Ready check for Seer initialization
 *
 * @returns {Boolean}
 */
var isReady = function isReady() {
  return isBrowser && window.__SEER_INITIALIZED__;
};

/**
 * Utility method allowing to throttle a user action based on a key and a minimun delay.
 *
 * @param key {String} A unique key
 * @param delay {Number} The minimal delay to throttle
 * @returns {Boolean}
 */
var throttle = function throttle(key, delay) {
  var time = timers.get(key);
  var now = Date.now();
  if (time && now - time < delay) {
    return true;
  }
  timers.set(key, now);
  return false;
};

var replacer = function replacer(seen) {
  return function (key, value) {
    if (value && (typeof value === 'undefined' ? 'undefined' : _typeof(value)) === 'object' && seen.has(value)) {
      return;
    }
    seen.add(value);
    var isArray = Object.prototype.toString.call(value).slice(8, -1).includes('Array');
    if (isArray) {
      return Array.prototype.slice.call(value, 0, 20);
    }
    return value;
  };
};

/**
 * Low-level api leveraging window.postMessage
 *
 * @param type {String} The action type
 * @param payload {Any} The action payload
 */
var send = function send(type) {
  var data = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  if (!isBrowser || !isReady()) {
    return;
  }

  var seen = new Set();
  var payload = JSON.stringify(data, replacer(seen));

  try {
    window.postMessage({ type: type, payload: payload, source: 'seer-agent' }, '*');
  } catch (e) {
    if (throttle('seer-log', 2E3)) {
      return;
    }
    console.log(e); // eslint-disable-line
  }
};

var listeners = new Map();

var listener = function listener(message) {
  if (!message || !message.data || message.data.source !== 'seer-core') {
    return;
  }
  var _message$data = message.data,
      type = _message$data.type,
      payload = _message$data.payload;


  var typeListeners = listeners.get(type);
  if (typeListeners) {
    typeListeners.forEach(function (cb) {
      return cb(payload);
    });
  }
};

/**
 * Initilize window listener. There will be only one for the whole process
 * to prevent too many registrations.
 *
 * This method will be called automatically if you use the `listenFor` method.
 */
var init = function init() {
  if (!isBrowser || window.__SEER_LISTENER__) {
    return;
  }
  window.addEventListener('message', listener);
  window.__SEER_LISTENER__ = true;
};

/**
 * Clean listener. Can be useful in case you want to unregister upcoming events
 * or liberate memory.
 */
var clean = function clean() {
  if (!isBrowser || !window.__SEER_LISTENER__) {
    return;
  }
  window.removeEventListener('message', listener);
  delete window.__SEER_LISTENER__;
};

/**
 * Create a listener that will be called upon events of the given key.
 *
 * @param key {String} The unique tab key
 * @param cb {Function} A callback that will receive the message payload
 */
var listenFor = function listenFor(type, cb) {
  if (!isBrowser) {
    return;
  }
  if (!type || !cb) {
    throw new Error('Please provide a type and callback');
  }
  if (!listeners.has(type)) {
    listeners.set(type, []);
  }
  if (!window.__SEER_LISTENER__) {
    init();
  }
  listeners.get(type).push(cb);
};

/**
 * Remove an identity listener
 *
 * @param cb {Function} The callback to remove
 */
var removeListener = function removeListener(cb) {
  listeners.forEach(function (typeListeners, key) {
    listeners.set(key, typeListeners.filter(function (l) {
      return l !== cb;
    }));
  });
};

/**
 * Creates a new indexed list.
 * It works by index to get O(1) accessing and performance.
 *
 * @param key {String} The key of the tab
 * @param data {Object} The indexed object
 */
var list = function list(key, data) {
  return send('LIST', { key: key, data: data });
};

/**
 * Creates an element in the indexed list, based on the itemKey.
 *
 * @param key {String} The key of the tab
 * @param itemKey {String} The key of the item
 * @param data {Any} The value of the item
 */
var listItem = function listItem(key, itemKey) {
  var data = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  return send('LIST_ITEM', { key: key, itemKey: itemKey, data: data });
};

/**
 * Update an item property, can be deeply nested.
 *
 * @param key {String} The key of the tab
 * @param itemKey {String} The key of the item
 * @param path {String} The path of the variable you want to update
 * @param data {Object} The new value
 */
var updateItem = function updateItem(key, itemKey, path, data) {
  return send('UPDATE_ITEM', { key: key, itemKey: itemKey, path: path, data: data });
};

/**
 * Similar to updateItem, but allows to pass an array with {path,data} pairs for
 * multiple update of the same item without having to send multiple messages.
 *
 * @param key {String} The key of the tab
 * @param itemKey {String} The key of the item
 * @param array {Array} The array of updates
 * @param array.path {String} The path for this update
 * @param array.data {Object} The value of this update
 */
var multiUpdate = function multiUpdate(key, itemKey, array) {
  return send('MULTI_UPDATE_ITEM', { key: key, itemKey: itemKey, array: array });
};

/**
 * Remove a specific item in a specific tab.
 *
 * @param key {String} They key of the tab
 * @param itemKey {String} The key of the item
 */
var deleteItem = function deleteItem(key, itemKey) {
  return send('DELETE_ITEM', { key: key, itemKey: itemKey });
};

/**
 * Will create a log message to an item, that will be displayde with the current time.
 *
 * @param key {String} The key of the tab
 * @param itemKey {String} The key of the item
 * @param msg {String} The message to display
 */
var addLog = function addLog(key, itemKey, msg) {
  return send('ADD_LOG', { key: key, itemKey: itemKey, msg: msg });
};

exports.default = {

  send: send,
  throttle: throttle,
  isReady: isReady,

  list: list,
  listItem: listItem,
  updateItem: updateItem,
  multiUpdate: multiUpdate,
  deleteItem: deleteItem,
  addLog: addLog,

  listeners: listeners,
  listenFor: listenFor,
  removeListener: removeListener,
  init: init,
  clean: clean

};
