"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const Debug = require("debug");
const path = require("path");
const tmp = require("tmp");
const docker_1 = require("../docker");
const debug = Debug("snyk");
function detect(targetImage, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const docker = new docker_1.Docker(targetImage, options);
        const info = yield docker.inspectImage(targetImage);
        return JSON.parse(info.stdout)[0];
    });
}
exports.detect = detect;
/**
 * In the case that an `ImageType.Identifier` is detected we need to produce
 * an image archive, either by saving the image if it's already loaded into
 * the local docker daemon, or by pulling the image from a remote registry and
 * saving it to the filesystem directly.
 *
 * Users may also provide us with a URL to an image in a Docker compatible
 * remote registry.
 *
 * @param {string} targetImage - The image to test, this could be in one of
 *    the following forms:
 *      * [registry/]<repo>/<image>[:tag]
 *      * <repo>/<image>[:tag]
 *      * <image>[:tag]
 *    In the case that a registry is not provided, the plugin will default
 *    this to Docker Hub. If a tag is not provided this will default to
 *    `latest`.
 * @param {string} [username] - Optional username for private repo auth.
 * @param {string} [password] - Optional password for private repo auth.
 */
function getImageArchive(targetImage, username, password) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const docker = new docker_1.Docker(targetImage);
        const destination = tmp.dirSync({ unsafeCleanup: true });
        const saveLocation = path.join(destination.name, "image.tar");
        try {
            // TODO: Eventually need to provide support for specifying a user-defined
            // temporary directory.
            yield docker.save(targetImage, saveLocation);
            return {
                path: saveLocation,
                removeArchive: destination.removeCallback,
            };
        }
        catch (_a) {
            debug(`${targetImage} does not exist locally, proceeding to pull image from remote registry.`);
        }
        if (yield docker_1.Docker.binaryExists()) {
            try {
                if (username || password) {
                    debug("using local docker binary credentials. the credentials you provided will be ignored");
                }
                yield docker.pullCli(targetImage);
                yield docker.save(targetImage, saveLocation);
                return {
                    path: saveLocation,
                    removeArchive: destination.removeCallback,
                };
            }
            catch (err) {
                debug(`couldn't pull ${targetImage} using docker binary: ${err}`);
            }
        }
        destination.removeCallback();
        const { hostname, imageName, tag } = yield extractImageDetails(targetImage);
        debug(`Attempting to pull: registry: ${hostname}, image: ${imageName}, tag: ${tag}`);
        const pullResult = yield docker.pull(hostname, imageName, tag, username, password);
        return {
            path: path.join(pullResult.stagingDir.name, "image.tar"),
            removeArchive: pullResult.stagingDir.removeCallback,
        };
    });
}
exports.getImageArchive = getImageArchive;
function extractImageDetails(targetImage) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let remainder;
        let hostname;
        let imageName;
        let tag;
        // We need to detect if the `targetImage` is part of a URL. Based on the Docker specification,
        // the hostname should contain a `.` or `:` before the first instance of a `/` otherwise the
        // default hostname will be used (registry-1.docker.io). ref: https://stackoverflow.com/a/37867949
        const i = targetImage.indexOf("/");
        if (i === -1 ||
            (!targetImage.substring(0, i).includes(".") &&
                !targetImage.substring(0, i).includes(":") &&
                targetImage.substring(0, i) !== "localhost")) {
            hostname = "registry-1.docker.io";
            remainder = targetImage;
            [imageName, tag] = remainder.split(":");
            imageName =
                imageName.indexOf("/") === -1 ? "library/" + imageName : imageName;
        }
        else {
            hostname = targetImage.substring(0, i);
            remainder = targetImage.substring(i + 1);
            [imageName, tag] = remainder.split(":");
        }
        // Assume the latest tag if no tag was found.
        tag = tag || "latest";
        return {
            hostname,
            imageName,
            tag,
        };
    });
}
exports.extractImageDetails = extractImageDetails;
function pullIfNotLocal(targetImage, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const docker = new docker_1.Docker(targetImage);
        try {
            yield docker.inspectImage(targetImage);
            return;
        }
        catch (err) {
            // image doesn't exist locally
        }
        yield docker.pullCli(targetImage);
    });
}
exports.pullIfNotLocal = pullIfNotLocal;
//# sourceMappingURL=image-inspector.js.map