"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const snyk_docker_pull_1 = require("@snyk/snyk-docker-pull");
const Debug = require("debug");
const Modem = require("docker-modem");
const event_loop_spinner_1 = require("event-loop-spinner");
const fs_1 = require("fs");
const minimatch = require("minimatch");
const fspath = require("path");
const lsu = require("./ls-utils");
const subProcess = require("./sub-process");
const SystemDirectories = ["dev", "proc", "sys"];
const debug = Debug("snyk");
class Docker {
    constructor(targetImage, options) {
        var _a;
        this.targetImage = targetImage;
        this.optionsList = Docker.createOptionsList(options);
        this.socketPath = ((_a = options) === null || _a === void 0 ? void 0 : _a.socketPath) ? options.socketPath
            : "/var/run/docker.sock";
    }
    static binaryExists() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                yield subProcess.execute("docker", ["version"]);
                return true;
            }
            catch (e) {
                return false;
            }
        });
    }
    static run(args, options) {
        return subProcess.execute("docker", [
            ...Docker.createOptionsList(options),
            ...args,
        ]);
    }
    static createOptionsList(options) {
        const opts = [];
        if (!options) {
            return opts;
        }
        if (options.host) {
            opts.push(`--host=${options.host}`);
        }
        if (options.tlscert) {
            opts.push(`--tlscert=${options.tlscert}`);
        }
        if (options.tlscacert) {
            opts.push(`--tlscacert=${options.tlscacert}`);
        }
        if (options.tlskey) {
            opts.push(`--tlskey=${options.tlskey}`);
        }
        if (options.tlsverify) {
            opts.push(`--tlsverify=${options.tlsverify}`);
        }
        return opts;
    }
    /**
     * Runs the command, catching any expected errors and returning them as normal
     * stderr/stdout result.
     */
    runSafe(cmd, args = [], 
    // no error is thrown if any of listed errors is found in stderr
    ignoreErrors = ["No such file", "not found"]) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.run(cmd, args);
            }
            catch (error) {
                const stderr = error.stderr;
                if (typeof stderr === "string") {
                    if (ignoreErrors.some((errMsg) => stderr.indexOf(errMsg) >= 0)) {
                        return { stdout: error.stdout, stderr };
                    }
                }
                throw error;
            }
        });
    }
    run(cmd, args = []) {
        return subProcess.execute("docker", [
            ...this.optionsList,
            "run",
            "--rm",
            "--entrypoint",
            '""',
            "--network",
            "none",
            this.targetImage,
            cmd,
            ...args,
        ]);
    }
    pull(registry, repo, tag, username, password) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const dockerPull = new snyk_docker_pull_1.DockerPull();
            const opt = {
                username,
                password,
                loadImage: false,
            };
            return yield dockerPull.pull(registry, repo, tag, opt);
        });
    }
    pullCli(targetImage) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return subProcess.execute("docker", ["pull", targetImage]);
        });
    }
    save(targetImage, destination) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const request = {
                path: `/images/${targetImage}/get?`,
                method: "GET",
                isStream: true,
                statusCodes: {
                    200: true,
                    400: "bad request",
                    404: "not found",
                    500: "server error",
                },
            };
            debug(`Docker.save: targetImage: ${targetImage}, destination: ${destination}`);
            const modem = new Modem({ socketPath: this.socketPath });
            return new Promise((resolve, reject) => {
                modem.dial(request, (err, stream) => {
                    if (err) {
                        return reject(err);
                    }
                    const writeStream = fs_1.createWriteStream(destination);
                    writeStream.on("error", (err) => {
                        reject(err);
                    });
                    writeStream.on("finish", () => {
                        resolve();
                    });
                    stream.on("error", (err) => {
                        reject(err);
                    });
                    stream.on("end", () => {
                        writeStream.end();
                    });
                    stream.pipe(writeStream);
                });
            });
        });
    }
    inspectImage(targetImage) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return subProcess.execute("docker", [
                ...this.optionsList,
                "inspect",
                targetImage,
            ]);
        });
    }
    catSafe(filename) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return this.runSafe("cat", [filename]);
        });
    }
    lsSafe(path, recursive) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let params = "-1ap";
            if (recursive) {
                params += "R";
            }
            const ignoreErrors = [
                "No such file",
                "file not found",
                "Permission denied",
            ];
            return this.runSafe("ls", [params, path], ignoreErrors);
        });
    }
    /**
     * Find files on a docker image according to a given list of glob expressions.
     */
    findGlobs(globs, exclusionGlobs = [], path = "/", recursive = true, excludeRootDirectories = SystemDirectories) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let root;
            const res = [];
            if (recursive && path === "/") {
                // When scanning from the root of a docker image we need to
                // exclude system files e.g. /proc, /sys, etc. to make the
                // operation less expensive.
                const outputRoot = yield this.lsSafe("/", false);
                root = lsu.parseLsOutput(outputRoot.stdout);
                for (const subdir of root.subDirs) {
                    if (excludeRootDirectories.includes(subdir.name)) {
                        continue;
                    }
                    const subdirOutput = yield this.lsSafe("/" + subdir.name, true);
                    const subdirRecursive = lsu.parseLsOutput(subdirOutput.stdout);
                    yield lsu.iterateFiles(subdirRecursive, (f) => {
                        f.path = "/" + subdir.name + f.path;
                    });
                    subdir.subDirs = subdirRecursive.subDirs;
                    subdir.files = subdirRecursive.files;
                }
            }
            else {
                const output = yield this.lsSafe(path, recursive);
                if (event_loop_spinner_1.eventLoopSpinner.isStarving()) {
                    yield event_loop_spinner_1.eventLoopSpinner.spin();
                }
                root = lsu.parseLsOutput(output.stdout);
            }
            yield lsu.iterateFiles(root, (f) => {
                const filepath = fspath.join(f.path, f.name);
                let exclude = false;
                for (const g of exclusionGlobs) {
                    if (!exclude && minimatch(filepath, g)) {
                        exclude = true;
                    }
                }
                if (!exclude) {
                    for (const g of globs) {
                        if (minimatch(filepath, g)) {
                            res.push(filepath);
                        }
                    }
                }
            });
            return res;
        });
    }
}
exports.Docker = Docker;
//# sourceMappingURL=docker.js.map