"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const fs = require("fs");
const path = require("path");
const image_inspector_1 = require("./analyzer/image-inspector");
const image_type_1 = require("./image-type");
const staticModule = require("./static");
const types_1 = require("./types");
function experimentalAnalysis(targetImage, dockerfileAnalysis, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        // assume Distroless scanning
        const imageType = image_type_1.getImageType(targetImage);
        switch (imageType) {
            case types_1.ImageType.DockerArchive:
            case types_1.ImageType.OciArchive:
                return localArchive(targetImage, imageType, dockerfileAnalysis, options);
            case types_1.ImageType.Identifier:
                return distroless(targetImage, dockerfileAnalysis, options);
            default:
                throw new Error("Unhandled image type for image " + targetImage);
        }
    });
}
exports.experimentalAnalysis = experimentalAnalysis;
function localArchive(targetImage, imageType, dockerfileAnalysis, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const archivePath = image_type_1.getArchivePath(targetImage);
        if (!fs.existsSync(archivePath)) {
            throw new Error("The provided archive path does not exist on the filesystem");
        }
        if (!fs.lstatSync(archivePath).isFile()) {
            throw new Error("The provided archive path is not a file");
        }
        // The target image becomes the base of the path, e.g. "archive.tar" for "/var/tmp/archive.tar"
        const imageIdentifier = path.basename(archivePath);
        return yield getStaticAnalysisResult(imageIdentifier, archivePath, dockerfileAnalysis, imageType, options["app-vulns"]);
    });
}
// experimental flow expected to be merged with the static analysis when ready
function distroless(targetImage, dockerfileAnalysis, options) {
    var _a, _b;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (staticModule.isRequestingStaticAnalysis(options)) {
            options.staticAnalysisOptions.distroless = true;
            return staticModule.analyzeStatically(targetImage, dockerfileAnalysis, options);
        }
        const archiveResult = yield image_inspector_1.getImageArchive(targetImage, (_a = options) === null || _a === void 0 ? void 0 : _a.username, (_b = options) === null || _b === void 0 ? void 0 : _b.password);
        try {
            return yield getStaticAnalysisResult(targetImage, archiveResult.path, dockerfileAnalysis, types_1.ImageType.DockerArchive, options["app-vulns"]);
        }
        finally {
            archiveResult.removeArchive();
        }
    });
}
exports.distroless = distroless;
function getStaticAnalysisResult(targetImage, archivePath, dockerfileAnalysis, imageType, appScan) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const scanningOptions = {
            staticAnalysisOptions: {
                imagePath: archivePath,
                imageType,
                distroless: true,
                appScan,
            },
        };
        return yield staticModule.analyzeStatically(targetImage, dockerfileAnalysis, scanningOptions);
    });
}
//# sourceMappingURL=experimental.js.map