"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const Debug = require("debug");
const fs_1 = require("fs");
const gunzip = require("gunzip-maybe");
const path_1 = require("path");
const tar_stream_1 = require("tar-stream");
const stream_utils_1 = require("../../stream-utils");
const layer_1 = require("../layer");
const debug = Debug("snyk");
/**
 * Retrieve the products of files content from the specified docker-archive.
 * @param dockerArchiveFilesystemPath Path to image file saved in docker-archive format.
 * @param extractActions Array of pattern-callbacks pairs.
 * @returns Array of extracted files products sorted by the reverse order of the layers from last to first.
 */
function extractArchive(dockerArchiveFilesystemPath, extractActions) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        return new Promise((resolve, reject) => {
            const tarExtractor = tar_stream_1.extract();
            const layers = {};
            let manifest;
            tarExtractor.on("entry", (header, stream, next) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (header.type === "file") {
                    if (isTarFile(header.name)) {
                        try {
                            layers[header.name] = yield layer_1.extractImageLayer(stream, extractActions);
                        }
                        catch (error) {
                            debug(`Error extracting layer content from: '${error}'`);
                            reject(new Error("Error reading tar archive"));
                        }
                    }
                    else if (isManifestFile(header.name)) {
                        manifest = yield getManifestFile(stream);
                    }
                }
                stream.resume(); // auto drain the stream
                next(); // ready for next entry
            }));
            tarExtractor.on("finish", () => {
                try {
                    resolve(getLayersContentAndArchiveManifest(manifest, layers));
                }
                catch (error) {
                    debug(`Error getting layers and manifest content from docker archive: '${error}'`);
                    reject(new Error("Invalid Docker archive"));
                }
            });
            tarExtractor.on("error", (error) => reject(error));
            fs_1.createReadStream(dockerArchiveFilesystemPath)
                .pipe(gunzip())
                .pipe(tarExtractor);
        });
    });
}
exports.extractArchive = extractArchive;
function getLayersContentAndArchiveManifest(manifest, layers) {
    // skip (ignore) non-existent layers
    // get the layers content without the name
    // reverse layers order from last to first
    const filteredLayers = manifest.Layers.filter((layersName) => layers[layersName])
        .map((layerName) => layers[layerName])
        .reverse();
    if (filteredLayers.length === 0) {
        throw new Error("We found no layers in the provided image");
    }
    return {
        layers: filteredLayers,
        manifest,
    };
}
/**
 * Note: consumes the stream.
 */
function getManifestFile(stream) {
    return stream_utils_1.streamToJson(stream).then((manifest) => {
        return manifest[0];
    });
}
function isManifestFile(name) {
    return name === "manifest.json";
}
function isTarFile(name) {
    // For both "docker save" and "skopeo copy" style archives the
    // layers are represented as tar archives whose names end in .tar.
    // For Docker this is "layer.tar", for Skopeo - "<sha256ofLayer>.tar".
    return path_1.basename(name).endsWith(".tar");
}
//# sourceMappingURL=layer.js.map