"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const gunzip = require("gunzip-maybe");
const path_1 = require("path");
const tar_stream_1 = require("tar-stream");
const callbacks_1 = require("./callbacks");
/**
 * Extract key files from the specified TAR stream.
 * @param layerTarStream image layer as a Readable TAR stream. Note: consumes the stream.
 * @param extractActions array of pattern, callbacks pairs
 * @returns extracted file products
 */
function extractImageLayer(layerTarStream, extractActions) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        return new Promise((resolve, reject) => {
            const result = {};
            const tarExtractor = tar_stream_1.extract();
            tarExtractor.on("entry", (headers, stream, next) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (headers.type === "file") {
                    const absoluteFileName = path_1.resolve("/", headers.name);
                    // TODO wouldn't it be simpler to first check
                    // if the filename matches any patterns?
                    const processedResult = yield extractFileAndProcess(absoluteFileName, stream, extractActions);
                    if (processedResult !== undefined) {
                        result[absoluteFileName] = processedResult;
                    }
                }
                stream.resume(); // auto drain the stream
                next(); // ready for next entry
            }));
            tarExtractor.on("finish", () => {
                // all layer level entries read
                resolve(result);
            });
            tarExtractor.on("error", (error) => reject(error));
            layerTarStream.pipe(gunzip()).pipe(tarExtractor);
        });
    });
}
exports.extractImageLayer = extractImageLayer;
/**
 * Note: consumes the stream.
 */
function extractFileAndProcess(fileName, fileStream, extractActions) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const matchedActions = extractActions.filter((action) => action.filePathMatches(fileName));
        if (matchedActions.length > 0) {
            return yield callbacks_1.applyCallbacks(matchedActions, fileStream);
        }
        return undefined;
    });
}
//# sourceMappingURL=layer.js.map