"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const Debug = require("debug");
const path = require("path");
const analyzer = require("./analyzer");
const dependency_tree_1 = require("./dependency-tree");
const docker_1 = require("./docker");
const dockerFile = require("./docker-file");
exports.dockerFile = dockerFile;
const errors_1 = require("./errors");
const experimental_1 = require("./experimental");
const docker_2 = require("./inputs/runtime/docker");
const parser_1 = require("./parser");
const response_builder_1 = require("./response-builder");
const staticUtil = require("./static");
const types_1 = require("./types");
exports.ScanType = types_1.ScanType;
const MAX_MANIFEST_FILES = 5;
const debug = Debug("snyk");
function inspect(root, targetFile, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const targetImage = root;
        const dockerfileAnalysis = yield dockerFile.readDockerfileAndAnalyse(targetFile);
        if (options && options.experimental) {
            return yield experimental_1.experimentalAnalysis(targetImage, dockerfileAnalysis, options);
        }
        if (staticUtil.isRequestingStaticAnalysis(options)) {
            return yield staticUtil.analyzeStatically(targetImage, dockerfileAnalysis, options);
        }
        return yield analyzeDynamically(targetImage, dockerfileAnalysis, getDynamicAnalysisOptions(options));
    });
}
exports.inspect = inspect;
function analyzeDynamically(targetImage, dockerfileAnalysis, analysisOptions) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const [runtime, dependencies, manifestFiles] = yield Promise.all([
            docker_2.getRuntime(analysisOptions),
            getDependencies(targetImage, dockerfileAnalysis, analysisOptions),
            getManifestFiles(targetImage, analysisOptions),
        ]);
        return response_builder_1.buildResponse(runtime, dependencies, dockerfileAnalysis, manifestFiles, // bug in typescript wrongly adds `undefined`
        analysisOptions);
    });
}
// TODO: return type should be "DynamicAnalysisOptions" or something that extends DockerOptions
function getDynamicAnalysisOptions(options) {
    return options
        ? {
            host: options.host,
            tlsverify: options.tlsverify,
            tlscert: options.tlscert,
            tlscacert: options.tlscacert,
            tlskey: options.tlskey,
            manifestGlobs: options.manifestGlobs,
            manifestExcludeGlobs: options.manifestExcludeGlobs,
        }
        : {};
}
function getDependencies(targetImage, dockerfileAnalysis, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            const output = yield analyzer.analyzeDynamically(targetImage, dockerfileAnalysis, options);
            const result = parser_1.parseAnalysisResults(targetImage, output);
            const pkg = dependency_tree_1.buildTree(targetImage, result.type, result.depInfosList, result.targetOS);
            return {
                package: pkg,
                packageManager: result.type,
                imageId: result.imageId,
                binaries: result.binaries,
                imageLayers: result.imageLayers,
            };
        }
        catch (error) {
            const analysisError = errors_1.tryGetAnalysisError(error, targetImage);
            throw analysisError;
        }
    });
}
function getManifestFiles(targetImage, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!options.manifestGlobs) {
            return [];
        }
        let excludeGlobs = [];
        if (options.manifestExcludeGlobs) {
            excludeGlobs = options.manifestExcludeGlobs;
        }
        const globs = options.manifestGlobs;
        const docker = new docker_1.Docker(targetImage, options);
        let files = yield docker.findGlobs(globs, excludeGlobs);
        // Limit the number of manifest files which we return
        // to avoid overwhelming the docker daemon with cat requests
        if (files.length > MAX_MANIFEST_FILES) {
            debug(`Found ${files.length} manifest files in total. Only keeping the first ${MAX_MANIFEST_FILES}.`);
            files = files.slice(0, MAX_MANIFEST_FILES);
        }
        const contents = yield Promise.all(files.map((f) => docker.catSafe(f)));
        return files
            .map((g, i) => {
            return {
                name: path.basename(g),
                path: path.dirname(g),
                contents: Buffer.from(contents[i].stdout),
            };
        })
            .filter((i) => i.contents.length > 0);
    });
}
//# sourceMappingURL=index.js.map