"use strict";
const _ = require("lodash");
const chalk_1 = require("chalk");
const snyk = require("../../../lib");
const config = require("../../../lib/config");
const is_ci_1 = require("../../../lib/is-ci");
const api_token_1 = require("../../../lib/api-token");
const common_1 = require("../../../lib/snyk-test/common");
const Debug = require("debug");
const detect_1 = require("../../../lib/detect");
const types_1 = require("../../commands/types");
const iac_test_result_1 = require("../../../lib/snyk-test/iac-test-result");
const package_managers_1 = require("../../../lib/package-managers");
const analytics = require("../../../lib/analytics");
const fail_on_error_ts_1 = require("../../../lib/errors/fail-on-error.ts");
const formatters_1 = require("./formatters");
const utils = require("./utils");
const iac_output_1 = require("./iac-output");
const ecosystems_1 = require("../../../lib/ecosystems");
const errors_1 = require("../../../lib/errors");
const is_multi_project_scan_1 = require("../../../lib/is-multi-project-scan");
const sarif_output_1 = require("./sarif-output");
const constants_1 = require("../../../lib/iac/constants");
const debug = Debug('snyk-test');
const SEPARATOR = '\n-------------------------------------------------------\n';
const showVulnPathsMapping = {
    false: 'none',
    none: 'none',
    true: 'some',
    some: 'some',
    all: 'all',
};
// TODO: avoid using `as any` whenever it's possible
async function test(...args) {
    const resultOptions = [];
    const results = [];
    let options = {};
    if (typeof args[args.length - 1] === 'object') {
        options = args.pop();
    }
    // populate with default path (cwd) if no path given
    if (args.length === 0) {
        args.unshift(process.cwd());
    }
    // org fallback to config unless specified
    options.org = options.org || config.org;
    // making `show-vulnerable-paths` 'some' by default.
    const svpSupplied = (options['show-vulnerable-paths'] || '').toLowerCase();
    options.showVulnPaths = showVulnPathsMapping[svpSupplied] || 'some';
    if (options.severityThreshold &&
        !validateSeverityThreshold(options.severityThreshold)) {
        return Promise.reject(new Error('INVALID_SEVERITY_THRESHOLD'));
    }
    if (options.failOn && !validateFailOn(options.failOn)) {
        const error = new fail_on_error_ts_1.FailOnError();
        return Promise.reject(chalk_1.default.red.bold(error.message));
    }
    try {
        api_token_1.apiTokenExists();
    }
    catch (err) {
        if (options.docker && api_token_1.getDockerToken()) {
            options.testDepGraphDockerEndpoint = '/docker-jwt/test-dep-graph';
            options.isDockerUser = true;
        }
        else {
            throw err;
        }
    }
    const ecosystem = ecosystems_1.getEcosystem(options);
    if (ecosystem) {
        try {
            const commandResult = await ecosystems_1.testEcosystem(ecosystem, args, options);
            return commandResult;
        }
        catch (error) {
            throw new Error(error);
        }
    }
    // Promise waterfall to test all other paths sequentially
    for (const path of args) {
        // Create a copy of the options so a specific test can
        // modify them i.e. add `options.file` etc. We'll need
        // these options later.
        const testOpts = _.cloneDeep(options);
        testOpts.path = path;
        testOpts.projectName = testOpts['project-name'];
        let res;
        try {
            res = await snyk.test(path, testOpts);
        }
        catch (error) {
            // Possible error cases:
            // - the test found some vulns. `error.message` is a
            // JSON-stringified
            //   test result.
            // - the flow failed, `error` is a real Error object.
            // - the flow failed, `error` is a number or string
            // describing the problem.
            //
            // To standardise this, make sure we use the best _object_ to
            // describe the error.
            if (error instanceof Error) {
                res = error;
            }
            else if (typeof error !== 'object') {
                res = new Error(error);
            }
            else {
                try {
                    res = JSON.parse(error.message);
                }
                catch (unused) {
                    res = error;
                }
            }
        }
        // Not all test results are arrays in order to be backwards compatible
        // with scripts that use a callback with test. Coerce results/errors to be arrays
        // and add the result options to each to be displayed
        const resArray = Array.isArray(res) ? res : [res];
        for (let i = 0; i < resArray.length; i++) {
            const pathWithOptionalProjectName = utils.getPathWithOptionalProjectName(path, resArray[i]);
            results.push(_.assign(resArray[i], { path: pathWithOptionalProjectName }));
            // currently testOpts are identical for each test result returned even if it's for multiple projects.
            // we want to return the project names, so will need to be crafty in a way that makes sense.
            if (!testOpts.projectNames) {
                resultOptions.push(testOpts);
            }
            else {
                resultOptions.push(_.assign(_.cloneDeep(testOpts), {
                    projectName: testOpts.projectNames[i],
                }));
            }
        }
    }
    const vulnerableResults = results.filter((res) => (res.vulnerabilities && res.vulnerabilities.length) ||
        (res.result &&
            res.result.cloudConfigResults &&
            res.result.cloudConfigResults.length));
    const errorResults = results.filter((res) => res instanceof Error);
    const notSuccess = errorResults.length > 0;
    const foundVulnerabilities = vulnerableResults.length > 0;
    // resultOptions is now an array of 1 or more options used for
    // the tests results is now an array of 1 or more test results
    // values depend on `options.json` value - string or object
    const errorMappedResults = !options.iac
        ? createErrorMappedResultsForJsonOutput(results)
        : results.map(iac_test_result_1.mapIacTestResult);
    // backwards compat - strip array IFF only one result
    const jsonData = errorMappedResults.length === 1
        ? errorMappedResults[0]
        : errorMappedResults;
    const { stdout: dataToSend, stringifiedData, stringifiedJsonData, stringifiedSarifData, } = extractDataToSendFromResults(results, jsonData, options);
    if (options.json || options.sarif) {
        // if all results are ok (.ok == true) then return the json
        if (errorMappedResults.every((res) => res.ok)) {
            return types_1.TestCommandResult.createJsonTestCommandResult(stringifiedData, stringifiedJsonData, stringifiedSarifData);
        }
        const err = new Error(stringifiedData);
        if (foundVulnerabilities) {
            if (options.failOn) {
                const fail = shouldFail(vulnerableResults, options.failOn);
                if (!fail) {
                    // return here to prevent failure
                    return types_1.TestCommandResult.createJsonTestCommandResult(stringifiedData, stringifiedJsonData, stringifiedSarifData);
                }
            }
            err.code = 'VULNS';
            const dataToSendNoVulns = dataToSend;
            delete dataToSendNoVulns.vulnerabilities;
            err.jsonNoVulns = dataToSendNoVulns;
        }
        err.json = stringifiedData;
        err.jsonStringifiedResults = stringifiedJsonData;
        err.sarifStringifiedResults = stringifiedSarifData;
        throw err;
    }
    const pinningSupported = results.find((res) => res.packageManager === 'pip');
    let response = results
        .map((result, i) => {
        resultOptions[i].pinningSupported = pinningSupported;
        return displayResult(results[i], resultOptions[i], result.foundProjectCount);
    })
        .join(`\n${SEPARATOR}`);
    if (notSuccess) {
        debug(`Failed to test ${errorResults.length} projects, errors:`);
        errorResults.forEach((err) => {
            const errString = err.stack ? err.stack.toString() : err.toString();
            debug('error: %s', errString);
        });
    }
    let summaryMessage = '';
    if (results.length > 1) {
        const projects = results.length === 1 ? 'project' : 'projects';
        summaryMessage =
            `\n\n\nTested ${results.length} ${projects}` +
                formatters_1.summariseVulnerableResults(vulnerableResults, options) +
                formatters_1.summariseErrorResults(errorResults.length) +
                '\n';
    }
    if (notSuccess) {
        response += chalk_1.default.bold.red(summaryMessage);
        const error = new Error(response);
        // take the code of the first problem to go through error
        // translation
        // HACK as there can be different errors, and we pass only the
        // first one
        error.code = errorResults[0].code;
        error.userMessage = errorResults[0].userMessage;
        if (error.userMessage === errors_1.TestLimitReachedError().userMessage &&
            options.isDockerUser) {
            error.userMessage =
                'You have reached your scan limit. Sign up to Snyk for additional free scans https://dockr.ly/3ePqVcp';
        }
        throw error;
    }
    if (foundVulnerabilities) {
        if (options.failOn) {
            const fail = shouldFail(vulnerableResults, options.failOn);
            if (!fail) {
                // return here to prevent throwing failure
                response += chalk_1.default.bold.green(summaryMessage);
                return types_1.TestCommandResult.createHumanReadableTestCommandResult(response, stringifiedJsonData, stringifiedSarifData);
            }
        }
        response += chalk_1.default.bold.red(summaryMessage);
        const error = new Error(response);
        // take the code of the first problem to go through error
        // translation
        // HACK as there can be different errors, and we pass only the
        // first one
        error.code = vulnerableResults[0].code || 'VULNS';
        error.userMessage = vulnerableResults[0].userMessage;
        error.jsonStringifiedResults = stringifiedJsonData;
        error.sarifStringifiedResults = stringifiedSarifData;
        throw error;
    }
    response += chalk_1.default.bold.green(summaryMessage);
    return types_1.TestCommandResult.createHumanReadableTestCommandResult(response, stringifiedJsonData, stringifiedSarifData);
}
function createErrorMappedResultsForJsonOutput(results) {
    const errorMappedResults = results.map((result) => {
        // add json for when thrown exception
        if (result instanceof Error) {
            return {
                ok: false,
                error: result.message,
                path: result.path,
            };
        }
        return result;
    });
    return errorMappedResults;
}
function shouldFail(vulnerableResults, failOn) {
    // find reasons not to fail
    if (failOn === 'all') {
        return hasFixes(vulnerableResults);
    }
    if (failOn === 'upgradable') {
        return hasUpgrades(vulnerableResults);
    }
    if (failOn === 'patchable') {
        return hasPatches(vulnerableResults);
    }
    // should fail by default when there are vulnerable results
    return vulnerableResults.length > 0;
}
function isFixable(testResult) {
    return isUpgradable(testResult) || isPatchable(testResult);
}
function hasFixes(testResults) {
    return testResults.some(isFixable);
}
function isUpgradable(testResult) {
    if (testResult.remediation) {
        const { remediation: { upgrade = {}, pin = {} }, } = testResult;
        return Object.keys(upgrade).length > 0 || Object.keys(pin).length > 0;
    }
    // if remediation is not available, fallback on vuln properties
    const { vulnerabilities = {} } = testResult;
    return vulnerabilities.some(isVulnUpgradable);
}
function hasUpgrades(testResults) {
    return testResults.some(isUpgradable);
}
function isPatchable(testResult) {
    if (testResult.remediation) {
        const { remediation: { patch = {} }, } = testResult;
        return Object.keys(patch).length > 0;
    }
    // if remediation is not available, fallback on vuln properties
    const { vulnerabilities = {} } = testResult;
    return vulnerabilities.some(isVulnPatchable);
}
function hasPatches(testResults) {
    return testResults.some(isPatchable);
}
function isVulnUpgradable(vuln) {
    return vuln.isUpgradable || vuln.isPinnable;
}
function isVulnPatchable(vuln) {
    return vuln.isPatchable;
}
function isVulnFixable(vuln) {
    return isVulnUpgradable(vuln) || isVulnPatchable(vuln);
}
function displayResult(res, options, foundProjectCount) {
    const meta = formatters_1.formatTestMeta(res, options);
    const dockerAdvice = formatters_1.dockerRemediationForDisplay(res);
    const projectType = res.packageManager || options.packageManager;
    const localPackageTest = detect_1.isLocalFolder(options.path);
    let testingPath = options.path;
    if (options.iac && options.file) {
        testingPath = options.file;
    }
    const prefix = chalk_1.default.bold.white('\nTesting ' + testingPath + '...\n\n');
    // handle errors by extracting their message
    if (res instanceof Error) {
        return prefix + res.message;
    }
    const issuesText = res.licensesPolicy ||
        constants_1.TEST_SUPPORTED_IAC_PROJECTS.includes(projectType)
        ? 'issues'
        : 'vulnerabilities';
    let pathOrDepsText = '';
    if (res.dependencyCount) {
        pathOrDepsText += res.dependencyCount + ' dependencies';
    }
    else {
        pathOrDepsText += options.path;
    }
    const testedInfoText = `Tested ${pathOrDepsText} for known ${issuesText}`;
    let multiProjAdvice = '';
    const advertiseGradleSubProjectsCount = projectType === 'gradle' &&
        !options['gradle-sub-project'] &&
        !options.allProjects &&
        foundProjectCount;
    if (advertiseGradleSubProjectsCount) {
        multiProjAdvice = chalk_1.default.bold.white(`\n\nTip: This project has multiple sub-projects (${foundProjectCount}), ` +
            'use --all-sub-projects flag to scan all sub-projects.');
    }
    const advertiseAllProjectsCount = projectType !== 'gradle' &&
        !is_multi_project_scan_1.isMultiProjectScan(options) &&
        foundProjectCount;
    if (advertiseAllProjectsCount) {
        multiProjAdvice = chalk_1.default.bold.white(`\n\nTip: Detected multiple supported manifests (${foundProjectCount}), ` +
            'use --all-projects to scan all of them at once.');
    }
    // OK  => no vulns found, return
    if (res.ok && res.vulnerabilities.length === 0) {
        const vulnPathsText = options.showVulnPaths
            ? 'no vulnerable paths found.'
            : 'none were found.';
        const summaryOKText = chalk_1.default.green(`✓ ${testedInfoText}, ${vulnPathsText}`);
        const nextStepsText = localPackageTest
            ? '\n\nNext steps:' +
                '\n- Run `snyk monitor` to be notified ' +
                'about new related vulnerabilities.' +
                '\n- Run `snyk test` as part of ' +
                'your CI/test.'
            : '';
        // user tested a package@version and got 0 vulns back, but there were dev deps
        // to consider
        const snykPackageTestTip = !(options.docker ||
            localPackageTest ||
            options.dev)
            ? '\n\nTip: Snyk only tests production dependencies by default. You can try re-running with the `--dev` flag.'
            : '';
        const dockerCTA = dockerUserCTA(options);
        return (prefix +
            meta +
            '\n\n' +
            summaryOKText +
            multiProjAdvice +
            (is_ci_1.isCI()
                ? ''
                : dockerAdvice + nextStepsText + snykPackageTestTip + dockerCTA));
    }
    if (constants_1.TEST_SUPPORTED_IAC_PROJECTS.includes(res.packageManager)) {
        return iac_output_1.getIacDisplayedOutput(res, testedInfoText, meta, prefix);
    }
    // NOT OK => We found some vulns, let's format the vulns info
    return getDisplayedOutput(res, options, testedInfoText, localPackageTest, projectType, meta, prefix, multiProjAdvice, dockerAdvice);
}
function getDisplayedOutput(res, options, testedInfoText, localPackageTest, projectType, meta, prefix, multiProjAdvice, dockerAdvice) {
    const vulnCount = res.vulnerabilities && res.vulnerabilities.length;
    const singleVulnText = res.licensesPolicy ? 'issue' : 'vulnerability';
    const multipleVulnsText = res.licensesPolicy ? 'issues' : 'vulnerabilities';
    // Text will look like so:
    // 'found 232 vulnerabilities, 404 vulnerable paths.'
    let vulnCountText = `found ${res.uniqueCount} ` +
        (res.uniqueCount === 1 ? singleVulnText : multipleVulnsText);
    // Docker is currently not supported as num of paths is inaccurate due to trimming of paths to reduce size.
    if (options.showVulnPaths && !options.docker) {
        vulnCountText += `, ${vulnCount} vulnerable `;
        if (vulnCount === 1) {
            vulnCountText += 'path.';
        }
        else {
            vulnCountText += 'paths.';
        }
    }
    else {
        vulnCountText += '.';
    }
    const reachableVulnsText = options.reachableVulns && vulnCount > 0
        ? ` ${formatters_1.summariseReachableVulns(res.vulnerabilities)}`
        : '';
    const summary = testedInfoText +
        ', ' +
        chalk_1.default.red.bold(vulnCountText) +
        chalk_1.default.blue.bold(reachableVulnsText);
    let wizardAdvice = '';
    if (localPackageTest &&
        package_managers_1.WIZARD_SUPPORTED_PACKAGE_MANAGERS.includes(projectType)) {
        wizardAdvice = chalk_1.default.bold.green('\n\nRun `snyk wizard` to address these issues.');
    }
    const dockerSuggestion = getDockerSuggestionText(options, config);
    const vulns = res.vulnerabilities || [];
    const groupedVulns = groupVulnerabilities(vulns);
    const sortedGroupedVulns = _.orderBy(groupedVulns, ['metadata.severityValue', 'metadata.name'], ['asc', 'desc']);
    const filteredSortedGroupedVulns = sortedGroupedVulns.filter((vuln) => vuln.metadata.packageManager !== 'upstream');
    const binariesSortedGroupedVulns = sortedGroupedVulns.filter((vuln) => vuln.metadata.packageManager === 'upstream');
    let groupedVulnInfoOutput;
    if (res.remediation) {
        analytics.add('actionableRemediation', true);
        groupedVulnInfoOutput = formatters_1.formatIssuesWithRemediation(filteredSortedGroupedVulns, res.remediation, options);
    }
    else {
        analytics.add('actionableRemediation', false);
        groupedVulnInfoOutput = filteredSortedGroupedVulns.map((vuln) => formatters_1.formatIssues(vuln, options));
    }
    const groupedDockerBinariesVulnInfoOutput = res.docker && binariesSortedGroupedVulns.length
        ? formatters_1.formatDockerBinariesIssues(binariesSortedGroupedVulns, res.docker.binariesVulns, options)
        : [];
    let body = groupedVulnInfoOutput.join('\n\n') +
        '\n\n' +
        groupedDockerBinariesVulnInfoOutput.join('\n\n') +
        '\n\n' +
        meta;
    if (res.remediation) {
        body = summary + body + wizardAdvice;
    }
    else {
        body = body + '\n\n' + summary + wizardAdvice;
    }
    const ignoredIssues = '';
    const dockerCTA = dockerUserCTA(options);
    return (prefix +
        body +
        multiProjAdvice +
        ignoredIssues +
        dockerAdvice +
        dockerSuggestion +
        dockerCTA);
}
function validateSeverityThreshold(severityThreshold) {
    return common_1.SEVERITIES.map((s) => s.verboseName).indexOf(severityThreshold) > -1;
}
function validateFailOn(arg) {
    return Object.keys(common_1.FAIL_ON).includes(arg);
}
function groupVulnerabilities(vulns) {
    return vulns.reduce((map, curr) => {
        if (!map[curr.id]) {
            map[curr.id] = {};
            map[curr.id].list = [];
            map[curr.id].metadata = metadataForVuln(curr);
            map[curr.id].isIgnored = false;
            map[curr.id].isPatched = false;
            // Extra added fields for ease of handling
            map[curr.id].title = curr.title;
            map[curr.id].note = curr.note;
            map[curr.id].severity = curr.severity;
            map[curr.id].originalSeverity = curr.originalSeverity;
            map[curr.id].isNew = isNewVuln(curr);
            map[curr.id].name = curr.name;
            map[curr.id].version = curr.version;
            map[curr.id].fixedIn = curr.fixedIn;
            map[curr.id].dockerfileInstruction = curr.dockerfileInstruction;
            map[curr.id].dockerBaseImage = curr.dockerBaseImage;
            map[curr.id].nearestFixedInVersion = curr.nearestFixedInVersion;
            map[curr.id].legalInstructionsArray = curr.legalInstructionsArray;
            map[curr.id].reachability = curr.reachability;
        }
        map[curr.id].list.push(curr);
        if (!map[curr.id].isFixable) {
            map[curr.id].isFixable = isVulnFixable(curr);
        }
        if (!map[curr.id].note) {
            map[curr.id].note = !!curr.note;
        }
        return map;
    }, {});
}
// check if vuln was published in the last month
function isNewVuln(vuln) {
    const MONTH = 30 * 24 * 60 * 60 * 1000;
    const publicationTime = new Date(vuln.publicationTime).getTime();
    return publicationTime > Date.now() - MONTH;
}
function metadataForVuln(vuln) {
    return {
        id: vuln.id,
        title: vuln.title,
        description: vuln.description,
        type: vuln.type,
        name: vuln.name,
        info: vuln.info,
        severity: vuln.severity,
        severityValue: formatters_1.getSeverityValue(vuln.severity),
        isNew: isNewVuln(vuln),
        version: vuln.version,
        packageManager: vuln.packageManager,
    };
}
function getDockerSuggestionText(options, config) {
    if (!options.docker || options.isDockerUser) {
        return '';
    }
    let dockerSuggestion = '';
    if (config && config.disableSuggestions !== 'true') {
        const optOutSuggestions = '\n\nTo remove this message in the future, please run `snyk config set disableSuggestions=true`';
        if (!options.file) {
            dockerSuggestion +=
                chalk_1.default.bold.white('\n\nPro tip: use `--file` option to get base image remediation advice.' +
                    `\nExample: $ snyk test --docker ${options.path} --file=path/to/Dockerfile`) + optOutSuggestions;
        }
        else if (!options['exclude-base-image-vulns']) {
            dockerSuggestion +=
                chalk_1.default.bold.white('\n\nPro tip: use `--exclude-base-image-vulns` to exclude from display Docker base image vulnerabilities.') + optOutSuggestions;
        }
    }
    return dockerSuggestion;
}
function dockerUserCTA(options) {
    if (options.isDockerUser) {
        return '\n\nFor more free scans that keep your images secure, sign up to Snyk at https://dockr.ly/3ePqVcp';
    }
    return '';
}
function extractDataToSendFromResults(results, jsonData, options) {
    let sarifData = {};
    if (options.sarif || options['sarif-file-output']) {
        sarifData = !options.iac
            ? sarif_output_1.createSarifOutputForContainers(results)
            : iac_output_1.createSarifOutputForIac(results);
    }
    const stringifiedJsonData = JSON.stringify(jsonData, null, 2);
    const stringifiedSarifData = JSON.stringify(sarifData, null, 2);
    const dataToSend = options.sarif ? sarifData : jsonData;
    const stringifiedData = options.sarif
        ? stringifiedSarifData
        : stringifiedJsonData;
    return {
        stdout: dataToSend,
        stringifiedData,
        stringifiedJsonData,
        stringifiedSarifData,
    };
}
module.exports = test;
//# sourceMappingURL=index.js.map