"use strict";
const url = require("url");
const open = require("open");
const uuid = require("uuid");
const Debug = require("debug");
const cli_spinner_1 = require("cli-spinner");
const snyk = require("../../../lib");
const is_authed_1 = require("./is-authed");
const is_ci_1 = require("../../../lib/is-ci");
const config = require("../../../lib/config");
const request = require("../../../lib/request");
const errors_1 = require("../../../lib/errors");
const errors_2 = require("../../../lib/errors");
const token_expired_error_1 = require("../../../lib/errors/token-expired-error");
const misconfigured_auth_in_ci_error_1 = require("../../../lib/errors/misconfigured-auth-in-ci-error");
const query_strings_1 = require("../../../lib/query-strings");
const apiUrl = url.parse(config.API);
const authUrl = apiUrl.protocol + '//' + apiUrl.host;
const debug = Debug('snyk-auth');
let attemptsLeft = 0;
function resetAttempts() {
    attemptsLeft = 30;
}
async function webAuth(via) {
    const token = uuid.v4(); // generate a random key
    const redirects = {
        wizard: '/authenticated',
    };
    let urlStr = authUrl + '/login?token=' + token;
    const utmParams = query_strings_1.getQueryParamsAsString();
    if (utmParams) {
        urlStr += '&' + utmParams;
    }
    // validate that via comes from our code, and not from user & CLI
    if (redirects[via]) {
        urlStr += '&redirectUri=' + Buffer.from(redirects[via]).toString('base64');
    }
    const msg = '\nNow redirecting you to our auth page, go ahead and log in,\n' +
        "and once the auth is complete, return to this prompt and you'll\n" +
        "be ready to start using snyk.\n\nIf you can't wait use this url:\n" +
        urlStr +
        '\n';
    // suppress this message in CI
    if (!is_ci_1.isCI()) {
        console.log(msg);
    }
    else {
        return Promise.reject(misconfigured_auth_in_ci_error_1.MisconfiguredAuthInCI());
    }
    const spinner = new cli_spinner_1.Spinner('Waiting...');
    spinner.setSpinnerString('|/-\\');
    const ipFamily = await getIpFamily();
    try {
        spinner.start();
        await setTimeout(() => {
            open(urlStr);
        }, 0);
        return await testAuthComplete(token, ipFamily);
    }
    finally {
        spinner.stop(true);
    }
}
async function testAuthComplete(token, ipFamily) {
    const payload = {
        body: {
            token,
        },
        url: config.API + '/verify/callback',
        json: true,
        method: 'post',
    };
    if (ipFamily) {
        payload.family = ipFamily;
    }
    return new Promise((resolve, reject) => {
        debug(payload);
        request(payload, (error, res, body) => {
            debug(error, (res || {}).statusCode, body);
            if (error) {
                return reject(error);
            }
            if (res.statusCode !== 200) {
                return reject(errorForFailedAuthAttempt(res, body));
            }
            // we have success
            if (body.api) {
                return resolve({
                    res,
                    body,
                });
            }
            // we need to wait and poll again in a moment
            setTimeout(() => {
                attemptsLeft--;
                if (attemptsLeft > 0) {
                    return resolve(testAuthComplete(token, ipFamily));
                }
                reject(token_expired_error_1.TokenExpiredError());
            }, 1000);
        });
    });
}
async function auth(apiToken, via) {
    let promise;
    resetAttempts();
    if (apiToken) {
        // user is manually setting the API token on the CLI - let's trust them
        promise = is_authed_1.verifyAPI(apiToken);
    }
    else {
        promise = webAuth(via);
    }
    return promise.then((data) => {
        const res = data.res;
        const body = res.body;
        debug(body);
        if (res.statusCode === 200 || res.statusCode === 201) {
            snyk.config.set('api', body.api);
            return ('\nYour account has been authenticated. Snyk is now ready to ' +
                'be used.\n');
        }
        throw errorForFailedAuthAttempt(res, body);
    });
}
/**
 * Resolve an appropriate error for a failed attempt to authenticate
 *
 * @param res The response from the API
 * @param body The body of the failed authentication request
 */
function errorForFailedAuthAttempt(res, body) {
    if (res.statusCode === 401 || res.statusCode === 403) {
        return errors_2.AuthFailedError(body.userMessage, res.statusCode);
    }
    else {
        const userMessage = body && body.userMessage;
        const error = new errors_1.CustomError(userMessage || 'Auth request failed');
        if (userMessage) {
            error.userMessage = userMessage;
        }
        error.code = res.statusCode;
        return error;
    }
}
async function getIpFamily() {
    const family = 6;
    try {
        // Dispatch a FORCED IPv6 request to test client's ISP and network capability
        await request({
            url: config.API + '/verify/callback',
            family,
            method: 'post',
        });
        return family;
    }
    catch (e) {
        return undefined;
    }
}
module.exports = auth;
//# sourceMappingURL=index.js.map