"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.zoomToScale = zoomToScale;
exports.scaleToZoom = scaleToZoom;
exports.lngLatToWorld = lngLatToWorld;
exports.worldToLngLat = worldToLngLat;
exports.getMeterZoom = getMeterZoom;
exports.getDistanceScales = getDistanceScales;
exports.getWorldPosition = getWorldPosition;
exports.getViewMatrix = getViewMatrix;
exports.getProjectionParameters = getProjectionParameters;
exports.getProjectionMatrix = getProjectionMatrix;
exports.worldToPixels = worldToPixels;
exports.pixelsToWorld = pixelsToWorld;

var _slicedToArray2 = _interopRequireDefault(require("@babel/runtime/helpers/slicedToArray"));

var _math = require("math.gl");

var _mathUtils = require("./math-utils");

var _perspective = _interopRequireDefault(require("gl-mat4/perspective"));

var _scale = _interopRequireDefault(require("gl-mat4/scale"));

var _translate = _interopRequireDefault(require("gl-mat4/translate"));

var _rotateX = _interopRequireDefault(require("gl-mat4/rotateX"));

var _rotateZ = _interopRequireDefault(require("gl-mat4/rotateZ"));

var _lerp = _interopRequireDefault(require("gl-vec2/lerp"));

var _assert = _interopRequireDefault(require("./assert"));

var PI = Math.PI;
var PI_4 = PI / 4;
var DEGREES_TO_RADIANS = PI / 180;
var RADIANS_TO_DEGREES = 180 / PI;
var TILE_SIZE = 512;
var EARTH_CIRCUMFERENCE = 40.03e6;
var DEFAULT_ALTITUDE = 1.5;

function zoomToScale(zoom) {
  return Math.pow(2, zoom);
}

function scaleToZoom(scale) {
  return Math.log2(scale);
}

function lngLatToWorld(_ref, scale) {
  var _ref2 = (0, _slicedToArray2.default)(_ref, 2),
      lng = _ref2[0],
      lat = _ref2[1];

  scale *= TILE_SIZE;
  var lambda2 = lng * DEGREES_TO_RADIANS;
  var phi2 = lat * DEGREES_TO_RADIANS;
  var x = scale * (lambda2 + PI) / (2 * PI);
  var y = scale * (PI - Math.log(Math.tan(PI_4 + phi2 * 0.5))) / (2 * PI);
  return [x, y];
}

function worldToLngLat(_ref3, scale) {
  var _ref4 = (0, _slicedToArray2.default)(_ref3, 2),
      x = _ref4[0],
      y = _ref4[1];

  scale *= TILE_SIZE;
  var lambda2 = x / scale * (2 * PI) - PI;
  var phi2 = 2 * (Math.atan(Math.exp(PI - y / scale * (2 * PI))) - PI_4);
  return [lambda2 * RADIANS_TO_DEGREES, phi2 * RADIANS_TO_DEGREES];
}

function getMeterZoom(_ref5) {
  var latitude = _ref5.latitude;
  (0, _assert.default)(Number.isFinite(latitude));
  var latCosine = Math.cos(latitude * DEGREES_TO_RADIANS);
  return scaleToZoom(EARTH_CIRCUMFERENCE * latCosine) - 9;
}

function getDistanceScales(_ref6) {
  var latitude = _ref6.latitude,
      longitude = _ref6.longitude,
      zoom = _ref6.zoom,
      scale = _ref6.scale,
      _ref6$highPrecision = _ref6.highPrecision,
      highPrecision = _ref6$highPrecision === void 0 ? false : _ref6$highPrecision;
  scale = scale !== undefined ? scale : zoomToScale(zoom);
  (0, _assert.default)(Number.isFinite(latitude) && Number.isFinite(longitude) && Number.isFinite(scale));
  var result = {};
  var worldSize = TILE_SIZE * scale;
  var latCosine = Math.cos(latitude * DEGREES_TO_RADIANS);
  var pixelsPerDegreeX = worldSize / 360;
  var pixelsPerDegreeY = pixelsPerDegreeX / latCosine;
  var altPixelsPerMeter = worldSize / EARTH_CIRCUMFERENCE / latCosine;
  result.pixelsPerMeter = [altPixelsPerMeter, altPixelsPerMeter, altPixelsPerMeter];
  result.metersPerPixel = [1 / altPixelsPerMeter, 1 / altPixelsPerMeter, 1 / altPixelsPerMeter];
  result.pixelsPerDegree = [pixelsPerDegreeX, pixelsPerDegreeY, altPixelsPerMeter];
  result.degreesPerPixel = [1 / pixelsPerDegreeX, 1 / pixelsPerDegreeY, 1 / altPixelsPerMeter];

  if (highPrecision) {
    var latCosine2 = DEGREES_TO_RADIANS * Math.tan(latitude * DEGREES_TO_RADIANS) / latCosine;
    var pixelsPerDegreeY2 = pixelsPerDegreeX * latCosine2 / 2;
    var altPixelsPerDegree2 = worldSize / EARTH_CIRCUMFERENCE * latCosine2;
    var altPixelsPerMeter2 = altPixelsPerDegree2 / pixelsPerDegreeY * altPixelsPerMeter;
    result.pixelsPerDegree2 = [0, pixelsPerDegreeY2, altPixelsPerDegree2];
    result.pixelsPerMeter2 = [altPixelsPerMeter2, 0, altPixelsPerMeter2];
  }

  return result;
}

function getWorldPosition(_ref7) {
  var longitude = _ref7.longitude,
      latitude = _ref7.latitude,
      zoom = _ref7.zoom,
      scale = _ref7.scale,
      meterOffset = _ref7.meterOffset,
      _ref7$distanceScales = _ref7.distanceScales,
      distanceScales = _ref7$distanceScales === void 0 ? null : _ref7$distanceScales;
  scale = scale !== undefined ? scale : zoomToScale(zoom);
  var center2d = lngLatToWorld([longitude, latitude], scale);
  var center = new _math.Vector3(center2d[0], center2d[1], 0);

  if (meterOffset) {
    distanceScales = distanceScales || getDistanceScales({
      latitude: latitude,
      longitude: longitude,
      scale: scale
    });
    var pixelPosition = new _math.Vector3(meterOffset).scale(distanceScales.pixelsPerMeter).scale([1, -1, 1]);
    center.add(pixelPosition);
  }

  return center;
}

function getViewMatrix(_ref8) {
  var height = _ref8.height,
      pitch = _ref8.pitch,
      bearing = _ref8.bearing,
      altitude = _ref8.altitude,
      _ref8$center = _ref8.center,
      center = _ref8$center === void 0 ? null : _ref8$center,
      _ref8$flipY = _ref8.flipY,
      flipY = _ref8$flipY === void 0 ? false : _ref8$flipY;
  var vm = (0, _mathUtils.createMat4)();
  (0, _translate.default)(vm, vm, [0, 0, -altitude]);
  (0, _scale.default)(vm, vm, [1, 1, 1 / height]);
  (0, _rotateX.default)(vm, vm, -pitch * DEGREES_TO_RADIANS);
  (0, _rotateZ.default)(vm, vm, bearing * DEGREES_TO_RADIANS);

  if (flipY) {
    (0, _scale.default)(vm, vm, [1, -1, 1]);
  }

  if (center) {
    (0, _translate.default)(vm, vm, new _math.Vector3(center).negate());
  }

  return vm;
}

function getProjectionParameters(_ref9) {
  var width = _ref9.width,
      height = _ref9.height,
      _ref9$altitude = _ref9.altitude,
      altitude = _ref9$altitude === void 0 ? DEFAULT_ALTITUDE : _ref9$altitude,
      _ref9$pitch = _ref9.pitch,
      pitch = _ref9$pitch === void 0 ? 0 : _ref9$pitch,
      _ref9$farZMultiplier = _ref9.farZMultiplier,
      farZMultiplier = _ref9$farZMultiplier === void 0 ? 1 : _ref9$farZMultiplier;
  var pitchRadians = pitch * DEGREES_TO_RADIANS;
  var halfFov = Math.atan(0.5 / altitude);
  var topHalfSurfaceDistance = Math.sin(halfFov) * altitude / Math.sin(Math.PI / 2 - pitchRadians - halfFov);
  var farZ = Math.cos(Math.PI / 2 - pitchRadians) * topHalfSurfaceDistance + altitude;
  return {
    fov: 2 * Math.atan(height / 2 / altitude),
    aspect: width / height,
    focalDistance: altitude,
    near: 0.1,
    far: farZ * farZMultiplier
  };
}

function getProjectionMatrix(_ref10) {
  var width = _ref10.width,
      height = _ref10.height,
      pitch = _ref10.pitch,
      altitude = _ref10.altitude,
      _ref10$farZMultiplier = _ref10.farZMultiplier,
      farZMultiplier = _ref10$farZMultiplier === void 0 ? 10 : _ref10$farZMultiplier;

  var _getProjectionParamet = getProjectionParameters({
    width: width,
    height: height,
    altitude: altitude,
    pitch: pitch,
    farZMultiplier: farZMultiplier
  }),
      fov = _getProjectionParamet.fov,
      aspect = _getProjectionParamet.aspect,
      near = _getProjectionParamet.near,
      far = _getProjectionParamet.far;

  var projectionMatrix = (0, _perspective.default)([], fov, aspect, near, far);
  return projectionMatrix;
}

function worldToPixels(xyz, pixelProjectionMatrix) {
  var _xyz = (0, _slicedToArray2.default)(xyz, 3),
      x = _xyz[0],
      y = _xyz[1],
      _xyz$ = _xyz[2],
      z = _xyz$ === void 0 ? 0 : _xyz$;

  (0, _assert.default)(Number.isFinite(x) && Number.isFinite(y) && Number.isFinite(z));
  return (0, _mathUtils.transformVector)(pixelProjectionMatrix, [x, y, z, 1]);
}

function pixelsToWorld(xyz, pixelUnprojectionMatrix) {
  var targetZ = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;

  var _xyz2 = (0, _slicedToArray2.default)(xyz, 3),
      x = _xyz2[0],
      y = _xyz2[1],
      z = _xyz2[2];

  (0, _assert.default)(Number.isFinite(x) && Number.isFinite(y));

  if (Number.isFinite(z)) {
    var coord = (0, _mathUtils.transformVector)(pixelUnprojectionMatrix, [x, y, z, 1]);
    return coord;
  }

  var coord0 = (0, _mathUtils.transformVector)(pixelUnprojectionMatrix, [x, y, 0, 1]);
  var coord1 = (0, _mathUtils.transformVector)(pixelUnprojectionMatrix, [x, y, 1, 1]);
  var z0 = coord0[2];
  var z1 = coord1[2];
  var t = z0 === z1 ? 0 : ((targetZ || 0) - z0) / (z1 - z0);
  return (0, _lerp.default)([], coord0, coord1, t);
}
//# sourceMappingURL=web-mercator-utils.js.map