import _slicedToArray from "@babel/runtime/helpers/esm/slicedToArray";
import { Vector3 } from 'math.gl';
import { createMat4, transformVector } from './math-utils';
import mat4_perspective from 'gl-mat4/perspective';
import mat4_scale from 'gl-mat4/scale';
import mat4_translate from 'gl-mat4/translate';
import mat4_rotateX from 'gl-mat4/rotateX';
import mat4_rotateZ from 'gl-mat4/rotateZ';
import vec2_lerp from 'gl-vec2/lerp';
import assert from './assert';
const PI = Math.PI;
const PI_4 = PI / 4;
const DEGREES_TO_RADIANS = PI / 180;
const RADIANS_TO_DEGREES = 180 / PI;
const TILE_SIZE = 512;
const EARTH_CIRCUMFERENCE = 40.03e6;
const DEFAULT_ALTITUDE = 1.5;
export function zoomToScale(zoom) {
  return Math.pow(2, zoom);
}
export function scaleToZoom(scale) {
  return Math.log2(scale);
}
export function lngLatToWorld([lng, lat], scale) {
  scale *= TILE_SIZE;
  const lambda2 = lng * DEGREES_TO_RADIANS;
  const phi2 = lat * DEGREES_TO_RADIANS;
  const x = scale * (lambda2 + PI) / (2 * PI);
  const y = scale * (PI - Math.log(Math.tan(PI_4 + phi2 * 0.5))) / (2 * PI);
  return [x, y];
}
export function worldToLngLat([x, y], scale) {
  scale *= TILE_SIZE;
  const lambda2 = x / scale * (2 * PI) - PI;
  const phi2 = 2 * (Math.atan(Math.exp(PI - y / scale * (2 * PI))) - PI_4);
  return [lambda2 * RADIANS_TO_DEGREES, phi2 * RADIANS_TO_DEGREES];
}
export function getMeterZoom({
  latitude
}) {
  assert(Number.isFinite(latitude));
  const latCosine = Math.cos(latitude * DEGREES_TO_RADIANS);
  return scaleToZoom(EARTH_CIRCUMFERENCE * latCosine) - 9;
}
export function getDistanceScales({
  latitude,
  longitude,
  zoom,
  scale,
  highPrecision = false
}) {
  scale = scale !== undefined ? scale : zoomToScale(zoom);
  assert(Number.isFinite(latitude) && Number.isFinite(longitude) && Number.isFinite(scale));
  const result = {};
  const worldSize = TILE_SIZE * scale;
  const latCosine = Math.cos(latitude * DEGREES_TO_RADIANS);
  const pixelsPerDegreeX = worldSize / 360;
  const pixelsPerDegreeY = pixelsPerDegreeX / latCosine;
  const altPixelsPerMeter = worldSize / EARTH_CIRCUMFERENCE / latCosine;
  result.pixelsPerMeter = [altPixelsPerMeter, altPixelsPerMeter, altPixelsPerMeter];
  result.metersPerPixel = [1 / altPixelsPerMeter, 1 / altPixelsPerMeter, 1 / altPixelsPerMeter];
  result.pixelsPerDegree = [pixelsPerDegreeX, pixelsPerDegreeY, altPixelsPerMeter];
  result.degreesPerPixel = [1 / pixelsPerDegreeX, 1 / pixelsPerDegreeY, 1 / altPixelsPerMeter];

  if (highPrecision) {
    const latCosine2 = DEGREES_TO_RADIANS * Math.tan(latitude * DEGREES_TO_RADIANS) / latCosine;
    const pixelsPerDegreeY2 = pixelsPerDegreeX * latCosine2 / 2;
    const altPixelsPerDegree2 = worldSize / EARTH_CIRCUMFERENCE * latCosine2;
    const altPixelsPerMeter2 = altPixelsPerDegree2 / pixelsPerDegreeY * altPixelsPerMeter;
    result.pixelsPerDegree2 = [0, pixelsPerDegreeY2, altPixelsPerDegree2];
    result.pixelsPerMeter2 = [altPixelsPerMeter2, 0, altPixelsPerMeter2];
  }

  return result;
}
export function getWorldPosition({
  longitude,
  latitude,
  zoom,
  scale,
  meterOffset,
  distanceScales = null
}) {
  scale = scale !== undefined ? scale : zoomToScale(zoom);
  const center2d = lngLatToWorld([longitude, latitude], scale);
  const center = new Vector3(center2d[0], center2d[1], 0);

  if (meterOffset) {
    distanceScales = distanceScales || getDistanceScales({
      latitude,
      longitude,
      scale
    });
    const pixelPosition = new Vector3(meterOffset).scale(distanceScales.pixelsPerMeter).scale([1, -1, 1]);
    center.add(pixelPosition);
  }

  return center;
}
export function getViewMatrix({
  height,
  pitch,
  bearing,
  altitude,
  center = null,
  flipY = false
}) {
  const vm = createMat4();
  mat4_translate(vm, vm, [0, 0, -altitude]);
  mat4_scale(vm, vm, [1, 1, 1 / height]);
  mat4_rotateX(vm, vm, -pitch * DEGREES_TO_RADIANS);
  mat4_rotateZ(vm, vm, bearing * DEGREES_TO_RADIANS);

  if (flipY) {
    mat4_scale(vm, vm, [1, -1, 1]);
  }

  if (center) {
    mat4_translate(vm, vm, new Vector3(center).negate());
  }

  return vm;
}
export function getProjectionParameters({
  width,
  height,
  altitude = DEFAULT_ALTITUDE,
  pitch = 0,
  farZMultiplier = 1
}) {
  const pitchRadians = pitch * DEGREES_TO_RADIANS;
  const halfFov = Math.atan(0.5 / altitude);
  const topHalfSurfaceDistance = Math.sin(halfFov) * altitude / Math.sin(Math.PI / 2 - pitchRadians - halfFov);
  const farZ = Math.cos(Math.PI / 2 - pitchRadians) * topHalfSurfaceDistance + altitude;
  return {
    fov: 2 * Math.atan(height / 2 / altitude),
    aspect: width / height,
    focalDistance: altitude,
    near: 0.1,
    far: farZ * farZMultiplier
  };
}
export function getProjectionMatrix({
  width,
  height,
  pitch,
  altitude,
  farZMultiplier = 10
}) {
  const _getProjectionParamet = getProjectionParameters({
    width,
    height,
    altitude,
    pitch,
    farZMultiplier
  }),
        fov = _getProjectionParamet.fov,
        aspect = _getProjectionParamet.aspect,
        near = _getProjectionParamet.near,
        far = _getProjectionParamet.far;

  const projectionMatrix = mat4_perspective([], fov, aspect, near, far);
  return projectionMatrix;
}
export function worldToPixels(xyz, pixelProjectionMatrix) {
  const _xyz = _slicedToArray(xyz, 3),
        x = _xyz[0],
        y = _xyz[1],
        _xyz$ = _xyz[2],
        z = _xyz$ === void 0 ? 0 : _xyz$;

  assert(Number.isFinite(x) && Number.isFinite(y) && Number.isFinite(z));
  return transformVector(pixelProjectionMatrix, [x, y, z, 1]);
}
export function pixelsToWorld(xyz, pixelUnprojectionMatrix, targetZ = 0) {
  const _xyz2 = _slicedToArray(xyz, 3),
        x = _xyz2[0],
        y = _xyz2[1],
        z = _xyz2[2];

  assert(Number.isFinite(x) && Number.isFinite(y));

  if (Number.isFinite(z)) {
    const coord = transformVector(pixelUnprojectionMatrix, [x, y, z, 1]);
    return coord;
  }

  const coord0 = transformVector(pixelUnprojectionMatrix, [x, y, 0, 1]);
  const coord1 = transformVector(pixelUnprojectionMatrix, [x, y, 1, 1]);
  const z0 = coord0[2];
  const z1 = coord1[2];
  const t = z0 === z1 ? 0 : ((targetZ || 0) - z0) / (z1 - z0);
  return vec2_lerp([], coord0, coord1, t);
}
//# sourceMappingURL=web-mercator-utils.js.map