import Viewport from './viewport';
import { zoomToScale, getWorldPosition, pixelsToWorld, lngLatToWorld, worldToLngLat, getProjectionMatrix, getViewMatrix } from './web-mercator-utils';
import fitBounds from './fit-bounds';
import vec2_add from 'gl-vec2/add';
import vec2_negate from 'gl-vec2/negate';
export default class WebMercatorViewport extends Viewport {
  constructor({
    width,
    height,
    latitude = 0,
    longitude = 0,
    zoom = 0,
    pitch = 0,
    bearing = 0,
    altitude = 1.5,
    farZMultiplier = 10
  } = {}) {
    width = width || 1;
    height = height || 1;
    const scale = zoomToScale(zoom);
    altitude = Math.max(0.75, altitude);
    const center = getWorldPosition({
      longitude,
      latitude,
      scale
    });
    const projectionMatrix = getProjectionMatrix({
      width,
      height,
      pitch,
      bearing,
      altitude,
      farZMultiplier
    });
    const viewMatrix = getViewMatrix({
      height,
      center,
      pitch,
      bearing,
      altitude,
      flipY: true
    });
    super({
      width,
      height,
      viewMatrix,
      projectionMatrix
    });
    this.latitude = latitude;
    this.longitude = longitude;
    this.zoom = zoom;
    this.pitch = pitch;
    this.bearing = bearing;
    this.altitude = altitude;
    this.scale = scale;
    this.center = center;
    Object.freeze(this);
  }

  projectFlat(lngLat, scale = this.scale) {
    return lngLatToWorld(lngLat, scale);
  }

  unprojectFlat(xy, scale = this.scale) {
    return worldToLngLat(xy, scale);
  }

  getMapCenterByLngLatPosition({
    lngLat,
    pos
  }) {
    const fromLocation = pixelsToWorld(pos, this.pixelUnprojectionMatrix);
    const toLocation = lngLatToWorld(lngLat, this.scale);
    const translate = vec2_add([], toLocation, vec2_negate([], fromLocation));
    const newCenter = vec2_add([], this.center, translate);
    return worldToLngLat(newCenter, this.scale);
  }

  getLocationAtPoint({
    lngLat,
    pos
  }) {
    return this.getMapCenterByLngLatPosition({
      lngLat,
      pos
    });
  }

  fitBounds(bounds, options = {}) {
    const width = this.width,
          height = this.height;

    const _fitBounds = fitBounds(Object.assign({
      width,
      height,
      bounds
    }, options)),
          longitude = _fitBounds.longitude,
          latitude = _fitBounds.latitude,
          zoom = _fitBounds.zoom;

    return new WebMercatorViewport({
      width,
      height,
      longitude,
      latitude,
      zoom
    });
  }

}
//# sourceMappingURL=web-mercator-viewport.js.map