import fs from 'fs-extra';
import { validate, object, array, string, number, boolean } from '@hapi/joi';
import Rx from 'rx';

const WAIT_FILE_SCHEMA =
/*#__PURE__*/
object().keys({
  resources:
  /*#__PURE__*/
  array().items(string().required()).required(),
  delay:
  /*#__PURE__*/
  number().integer().min(0).default(0),
  interval:
  /*#__PURE__*/
  number().integer().min(0).default(250),
  log:
  /*#__PURE__*/
  boolean().default(false),
  reverse:
  /*#__PURE__*/
  boolean().default(false),
  timeout:
  /*#__PURE__*/
  number().integer().min(0).default(Infinity),
  verbose:
  /*#__PURE__*/
  boolean().default(false),
  window:
  /*#__PURE__*/
  number().integer().min(0).default(750)
});

function waitFileImpl(oldOpts, cb) {
  const validResult = validate(oldOpts, WAIT_FILE_SCHEMA);

  if (validResult.error && cb) {
    return cb(validResult.error);
  }

  const opts = validResult.value; // it needs to be at least interval

  if (opts.window < opts.interval) {
    opts.window = opts.interval;
  }

  const output = opts.verbose ? console.log.bind(console) : function () {};
  const log = opts.log ? console.log.bind(console) : function () {}; // the resources last known to be waiting for

  let lastWaitForOutput;
  let timeoutTimer = null;

  if (opts.timeout !== Infinity) {
    timeoutTimer = setTimeout(function () {
      log('wait-file(%s) timed out waiting for: %s; exiting with error', process.pid, lastWaitForOutput);
      cb && cb(new Error('Timeout'));
    }, opts.timeout);
  }

  function cleanup(err) {
    if (timeoutTimer) {
      clearTimeout(timeoutTimer);
      timeoutTimer = null;
    }

    if (cb) {
      cb(err);
      cb = null;
    }
  }

  function createFile(file) {
    const fstat = Rx.Observable.fromNodeCallback(fs.stat);
    const source = fstat(file);
    const fakeStat = Rx.Observable.just({
      size: -1
    });
    return Rx.Observable.catch(source, fakeStat).map(function (stat) {
      return {
        val: stat.size,
        data: stat
      };
    });
  }
  /* Stability checking occurs by using an Rx window,
     It waits until all of the vals from the resources are >=0,
     then it waits for a window which has no changes
     (duplicate outputs are filtered by distinctUntilChanged)
  */


  let lastValues = null;
  const src = Rx.Observable.timer(opts.delay, opts.interval).concatMap(() => {
    return Rx.Observable.from(opts.resources).concatMap(function (resource) {
      return createFile(resource);
    }, function (resource, obj) {
      return {
        resource: resource,
        val: obj.val,
        data: obj.data
      };
    }).reduce(function (acc, x) {
      acc[x.resource] = x.val;
      return acc;
    }, {});
  }).map(function (values) {
    lastValues = values; // save lastValues for later ref

    return values;
  }).distinctUntilChanged().windowWithTime(opts.window);

  function lastValuesAllAvailable() {
    if (!lastValues) {
      return false;
    }

    var notReady = opts.resources.filter(function (k) {
      var lastValue = lastValues && lastValues[k];
      var result = typeof lastValue !== 'number' || lastValue < 0;
      return opts.reverse ? !result : result;
    }); // only output when changes

    var notReadyString = notReady.join(', ');

    if (notReadyString && notReadyString !== lastWaitForOutput) {
      log('wait-file(%s) waiting for: %s', process.pid, notReadyString);
      lastWaitForOutput = notReadyString;
    }

    return !notReady.length;
  }

  let subsc = src.subscribe(function (child) {
    let childSub = child.toArray().subscribe(function (x) {
      output('child next', x);

      if (lastValuesAllAvailable() && !x.length) {
        output('stabilized');
        log('wait-file(%s) exiting successfully found all: %s', process.pid, opts.resources.join(', '));
        childSub.dispose();
        subsc.dispose();
        cleanup();
      }
    }, function (err) {
      output('child err', err);
    }, function () {
      output('child complete');
    });
  }, function (err) {
    output('err: ', err);
    log('wait-file(%s) exiting with error', process.pid, err);
    cleanup(err);
  }, function () {
    output('complete');
    cleanup();
  });
}

function waitFile(opts, cb) {
  if (cb && cb !== undefined) {
    return waitFileImpl(opts, cb);
  } else {
    return new Promise((resolve, reject) => {
      waitFileImpl(opts, err => {
        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }
}

export { waitFile };
//# sourceMappingURL=wait-file.esm.js.map
