# standard imports
import sys
import time
import socket
import signal
import os
import logging
import stat
import argparse
import uuid

# external imports
from chaind import Environment
import confini
from hexathon import strip_0x
from chainlib.chain import ChainSpec
from chainlib.eth.connection import EthHTTPConnection
from chainlib.eth.block import block_latest
from chainsyncer.driver import HeadSyncer
from chainsyncer.db import dsn_from_config
from chainsyncer.db.models.base import SessionBase
from chainsyncer.backend.sql import SQLBackend
from chainsyncer.error import SyncDone

logging.basicConfig(level=logging.WARNING)
logg = logging.getLogger()

env = Environment(domain='eth', env=os.environ)

argparser = argparse.ArgumentParser('chainqueue transaction submission and trigger server')
argparser.add_argument('-c', '--config', dest='c', type=str, default=env.config_dir, help='configuration directory')
argparser.add_argument('-p', type=str, help='rpc endpoint')
argparser.add_argument('-i', type=str, help='chain spec')
argparser.add_argument('--runtime-dir', dest='runtime_dir', type=str, default=env.runtime_dir, help='runtime directory')
argparser.add_argument('--data-dir', dest='data_dir', type=str, default=env.data_dir, help='data directory')
argparser.add_argument('--session-id', dest='session_id', type=str, default=env.session, help='session id to use for session')
argparser.add_argument('--offset', type=int, default=None, help='block number to start sync')
argparser.add_argument('--interval', type=int, default=5, help='sync pool interval, in seconds')
argparser.add_argument('-v', action='store_true', help='be verbose')
argparser.add_argument('-vv', action='store_true', help='be very verbose')
args = argparser.parse_args(sys.argv[1:])

if args.vv:
    logg.setLevel(logging.DEBUG)
elif args.v:
    logg.setLevel(logging.INFO)

# process config
config = confini.Config(args.c)
config.process()
args_override = {
            'SESSION_RUNTIME_DIR': getattr(args, 'runtime_dir'),
            'SESSION_CHAIN_SPEC': getattr(args, 'i'),
            'RPC_ENDPOINT': getattr(args, 'p'),
            'SESSION_DATA_DIR': getattr(args, 'data_dir'),
            'SYNCER_LOOP_INTERVAL': getattr(args, 'interval'),
        }
config.dict_override(args_override, 'cli flag')

if config.get('DATABASE_ENGINE') == 'sqlite':
    config.add(os.path.join(config.get('SESSION_DATA_DIR'), config.get('DATABASE_NAME')), 'DATABASE_NAME', True)
 
config.add(getattr(args, 'session_id'), '_SESSION_ID', True)
config.add(getattr(args, 'offset'), '_SYNCER_BLOCK_OFFSET', True)
config.censor('PASSWORD', 'DATABASE')
logg.debug('config loaded\n{}'.format(config))

chain_spec = ChainSpec.from_chain_str(config.get('SESSION_CHAIN_SPEC'))

dsn = dsn_from_config(config)
SQLBackend.setup(dsn, debug=config.true('DATABASE_DEBUG'))
rpc = EthHTTPConnection(url=config.get('RPC_ENDPOINT'), chain_spec=chain_spec)


def main():
    if config.get('_SYNCER_BLOCK_OFFSET') == None:
        o = block_latest()
        r = rpc.do(o)
        config.add(r[1], '_SYNCER_BLOCK_OFFSET')

    syncer_backend = SQLBackend.live(chain_spec, 0)
    syncer = HeadSyncer(syncer_backend)
    #fltr = NoopFilter()
    #syncer.add_filter(fltr)

    try:
        logg.debug('block offset {}'.format(config.get('_SYNCER_BLOCK_OFFSET')))
        syncer.loop(int(config.get('SYNCER_LOOP_INTERVAL')), rpc)
    except SyncDone as e:
        sys.stderr.write("sync '{}' done at block {}\n".format(args.mode, e))

    sys.exit(0)


if __name__ == '__main__':
    main()
