# SPDX-License-Identifier: GPL-3.0-or-later

# standard imports
import io
import sys
import os
import json
import argparse
import logging
import urllib

# external imports
import chainlib.cli
from crypto_dev_signer.eth.signer import ReferenceSigner as EIP155Signer
from crypto_dev_signer.keystore.dict import DictKeystore
from hexathon import (
        add_0x,
        strip_0x,
        )

# local imports
from chainlib.eth.address import to_checksum
from chainlib.eth.connection import EthHTTPConnection
from chainlib.jsonrpc import (
        JSONRPCRequest,
        IntSequenceGenerator,
        )
from chainlib.eth.nonce import (
        RPCNonceOracle,
        OverrideNonceOracle,
        )
from chainlib.eth.gas import (
        RPCGasOracle,
        OverrideGasOracle,
        )
from chainlib.eth.tx import (
        TxFactory,
        raw,
        )
from chainlib.chain import ChainSpec
from chainlib.eth.runnable.util import decode_for_puny_humans

logging.basicConfig(level=logging.WARNING)
logg = logging.getLogger()

script_dir = os.path.dirname(os.path.realpath(__file__)) 
config_dir = os.path.join(script_dir, '..', 'data', 'config')

arg_flags = chainlib.cli.argflag_std_write
argparser = chainlib.cli.ArgumentParser(arg_flags)
argparser.add_argument('--local', action='store_true', help='Execute contract read call')
argparser.add_positional('data', type=str, help='Transaction data')
args = argparser.parse_args()
config = chainlib.cli.Config.from_args(args, arg_flags, extra_args={'local': None}, default_config_dir=config_dir)

block_all = args.ww
block_last = args.w or block_all

wallet = chainlib.cli.Wallet(EIP155Signer)
wallet.from_config(config)

rpc = chainlib.cli.Rpc(EthHTTPConnection, wallet=wallet)
conn = rpc.connect_by_config(config)

send = config.true('_RPC_SEND')

local = config.get('_LOCAL')
if local:
    send = False

chain_spec = ChainSpec.from_chain_str(config.get('CHAIN_SPEC'))

def main():

    signer = rpc.get_signer()
    signer_address = rpc.get_signer_address()

    recipient = None
    if config.get('_RECIPIENT') != None:
        recipient = add_0x(to_checksum(config.get('_RECIPIENT')))
        if not args.u and recipient != add_0x(recipient):
            raise ValueError('invalid checksum address')

    if local:
        j = JSONRPCRequest(id_generator=rpc.id_generator)
        o = j.template()
        o['method'] = 'eth_call'
        o['params'].append({
            'to': recipient,
            'from': signer_address,
            'value': '0x00',
            'gas': add_0x(int.to_bytes(8000000, 8, byteorder='big').hex()), # TODO: better get of network gas limit
            'gasPrice': '0x01',
            'data': add_0x(args.data),
            })
        o['params'].append('latest')
        o = j.finalize(o)
        r = conn.do(o)
        print(strip_0x(r))
        return

    elif signer_address != None:
        g = TxFactory(chain_spec, signer=rpc.get_signer(), gas_oracle=rpc.get_gas_oracle(), nonce_oracle=rpc.get_nonce_oracle())
        tx = g.template(signer_address, recipient, use_nonce=True)
        if args.data != None:
            tx = g.set_code(tx, add_0x(args.data))

        (tx_hash_hex, o) = g.finalize(tx, id_generator=rpc.id_generator)
   
        if send:
            r = conn.do(o)
            print(r)
        else:
            print(o)
            print(tx_hash_hex)

    else:
        o = raw(args.data, id_generator=rpc.id_generator)
        if send:
            r = conn.do(o)
            print(r)
        else:
            print(o)


if __name__ == '__main__':
    main()
