# standard imports
import re
import logging
import json
import datetime
import time
import enum
import socket
from urllib.request import (
        Request,
        urlopen,
        )
from urllib.parse import urlparse

# third-party imports
from hexathon import (
        add_0x,
        strip_0x,
        )

# local imports
from .error import (
        DefaultErrorParser,
        RevertEthException,
        )
from .rpc import (
        jsonrpc_template,
        jsonrpc_result,
        )

error_parser = DefaultErrorParser()
logg = logging.getLogger(__name__)


class ConnType(enum.Enum):

    HTTP = 0x100
    HTTP_SSL = 0x101
    WEBSOCKET = 0x200
    WEBSOCKET_SSL = 0x201
    UNIX = 0x1000


re_http = '^http(s)?://'
re_ws = '^ws(s)?://'
re_unix = '^ipc://'

def str_to_connspec(s):
    m = re.match(re_http, s)
    if m != None:
        if re.group(1) != None:
            return ConnType.HTTP_SSL
        return ConnType.HTTP

    m = re.match(re_ws, s)
    if m != None:
        if re.group(1) != None:
            return ConnType.WEBSOCKET_SSL
        return ConnType.WEBSOCKET


    m = re.match(re_unix, s)
    if m != None:
        return ConnType.UNIX

    raise ValueError('unknown connection type {}'.format(s))


def from_conntype(t):
    if t in [ConnType.HTTP, ConnType.HTTP_SSL]:
        return HTTPConnection
    elif t in [ConnType.UNIX]:
        return UnixConnection
    raise NotImplementedError(t)



class RPCConnection():

    __locations = {}

    def __init__(self, location=None):
        logg.debug('creating connection {}'.format(location))
        self.location = location

 
    @staticmethod
    def register_location(tag, location):
        conntype = str_to_connspec(location)
        u = urlparse(location)
        RPCConnection.__locations[tag] = (conntype, u.path)
        logg.info('registered rpc connection {} as {}'.format(location, conntype))


    @staticmethod
    def connect(tag):
        c = RPCConnection.__locations[tag]
        constructor = from_conntype(c[0])
        return constructor(c[1])


class HTTPConnection(RPCConnection):


    def do(self, o, error_parser=error_parser):
        req = Request(
                self.url,
                method='POST',
                )
        req.add_header('Content-Type', 'application/json')
        data = json.dumps(o)
        logg.debug('(HTTP) send {}'.format(data))
        r = urlopen(req, data=data.encode('utf-8'))
        logg.debug('(HTTP) recv {}'.format(r))
        result = json.load(r)
        if o['id'] != result['id']:
            raise ValueError('RPC id mismatch; sent {} received {}'.format(o['id'], result['id']))
        return jsonrpc_result(o, error_parser)

    
    def wait(self, tx_hash_hex, delay=0.5, timeout=0.0):
        t = datetime.datetime.utcnow()
        i = 0
        while True:
            o = jsonrpc_template()
            o['method'] ='eth_getTransactionReceipt'
            o['params'].append(add_0x(tx_hash_hex))
            req = Request(
                    self.url,
                    method='POST',
                    )
            req.add_header('Content-Type', 'application/json')
            data = json.dumps(o)
            logg.debug('(HTTP) receipt attempt {} {}'.format(i, data))
            res = urlopen(req, data=data.encode('utf-8'))
            r = json.load(res)

            e = jsonrpc_result(r, error_parser)
            if e != None:
                logg.debug('e {}'.format(strip_0x(e['status'])))
                if strip_0x(e['status']) == '00':
                    raise RevertEthException(tx_hash_hex)
                return e

            if timeout > 0.0:
                delta = (datetime.datetime.utcnow() - t) + datetime.timedelta(seconds=delay)
                if  delta.total_seconds() >= timeout:
                    raise TimeoutError(tx_hash)

            time.sleep(delay)
            i += 1

  
class UnixConnection(RPCConnection):


    def do(self, o, error_parser=error_parser):
        conn = socket.socket(family=socket.AF_UNIX, type=socket.SOCK_STREAM, proto=0)
        conn.connect(self.location)
        data = json.dumps(o)

        logg.debug('unix socket send {}'.format(data))
        l = len(data)
        n = 0
        while n < l:
            c = conn.send(data.encode('utf-8'))
            if c == 0:
                s.close()
                raise IOError('unix socket ({}/{}) {}'.format(n, l, data))
            n += c
        r = b''
        while True:
            b = conn.recv(4096)
            if len(b) == 0:
                break
            r += b
        conn.close()
        logg.debug('unix socket recv {}'.format(r.decode('utf-8')))
        result = json.loads(r)
        if result['id'] != o['id']:
            raise ValueError('RPC id mismatch; sent {} received {}'.format(o['id'], result['id']))

        return jsonrpc_result(result, error_parser)


    def wait(self, tx_hash_hex, delay=0.5, timeout=0.0):
        raise NotImplementedError('Not yet implemented for unix socket')
