# third-party imports
from hexathon import (
        add_0x,
        strip_0x,
        )
from crypto_dev_signer.eth.transaction import EIP155Transaction

# local imports
from chainlib.hash import keccak256_hex_to_hex
from chainlib.jsonrpc import jsonrpc_template
from chainlib.eth.tx import (
        TxFactory,
        TxFormat,
        raw,
        )
from chainlib.eth.constant import (
        MINIMUM_FEE_UNITS,
    )


def price():
    o = jsonrpc_template()
    o['method'] = 'eth_gasPrice'
    return o


def balance(address):
    o = jsonrpc_template()
    o['method'] = 'eth_getBalance'
    o['params'].append(address)
    return o


class Gas(TxFactory):

    def create(self, sender_address, recipient_address, value, tx_format=TxFormat.JSONRPC):
        tx = self.template(sender_address, recipient_address, use_nonce=True)
        tx['value'] = value
        txe = EIP155Transaction(tx, tx['nonce'], tx['chainId'])
        self.signer.signTransaction(txe)
        tx_raw = txe.rlp_serialize()
        tx_raw_hex = add_0x(tx_raw.hex())
        tx_hash_hex = add_0x(keccak256_hex_to_hex(tx_raw_hex))

        if tx_format == TxFormat.JSONRPC:
            o = raw(tx_raw_hex)
        elif tx_format == TxFormat.RLP_SIGNED:
            o = tx_raw_hex

        return (tx_hash_hex, o)


class RPCGasOracle:

    def __init__(self, conn, code_callback=None):
        self.conn = conn
        self.code_callback = code_callback


    def get_gas(self, code=None):
        o = price()
        r = self.conn.do(o)
        n = strip_0x(r)
        fee_units = MINIMUM_FEE_UNITS
        if self.code_callback != None:
            fee_units = self.code_callback(code)
        return (int(n, 16), fee_units)


class OverrideGasOracle:

    def __init__(self, price, limit=None):
        if limit == None:
            limit = MINIMUM_FEE_UNITS
        self.limit = limit
        self.price = price


    def get_gas(self, code=None):
        return (self.price, self.limit)


DefaultGasOracle = RPCGasOracle
