# standard imports
import logging
import tempfile
import base64

# extended imports
import gnupg
from http_hoba_auth import Hoba

# local imports
from cic_auth.client.base import (
        ClientSession,
        challenge_nonce,
    )
from cic_auth.client.util import get_auth_header_parts

logg = logging.getLogger(__name__)


class PGPClientSession(ClientSession):

    def __init__(self, origin, private_key=None, passphrase=None, gpg_dir=None):
        super(PGPClientSession, self).__init__(origin)

        actual_gpg_dir = gpg_dir
        if actual_gpg_dir == None:
            actual_gpg_dir = tempfile.mkdtemp()
        logg.info('using gpg dir {}'.format(actual_gpg_dir))
        self.gpg = gnupg.GPG(gnupghome=actual_gpg_dir)
        if private_key != None:
            import_result = self.gpg.import_keys(private_key)
            self.fingerprint = import_result.results[0]['fingerprint']
            if import_result.sec_read == 0:
                raise ValueError('Export bundle contained no private keys')
            elif gpg_dir == None and import_result.sec_imported > 1:
                logg.warning('multiple private keys found. key with fingerprint {} will be used to sign challenges'.format(self.fingerprint))
        else:
            NotImplementedError('currently only works with passed private key export blobs')
        self.passphrase = passphrase


    def hoba_kid(self, encoding='base64'):
        kid = None
        #if encoding == 'base64':
        #    b = bytes.fromhex(self.fingerprint)
        #    kid = base64.b64encode(b)
        kid = bytes.fromhex(self.fingerprint)
        return kid 
        
 
    def process_auth_challenge(self, header, encoding='base64', method='HOBA'):
        if method != 'HOBA':
            logg.error('only HOBA implemented for pgp handler client, got {}'.format(method))
            return None
        (method, o) = get_auth_header_parts(header)
        c = o['challenge']
        if encoding == 'base64':
            c = base64.b64decode(o['challenge'])
        elif encoding != None:
            NotImplementedError(encoding)
       
        hoba = Hoba(self.origin, o['realm'])
        hoba.challenge = c
        hoba.nonce = challenge_nonce()
        hoba.kid = self.hoba_kid()
        hoba.alg = '969'
        
        plaintext = hoba.to_be_signed()

        r = self.gpg.sign(plaintext, passphrase=self.passphrase, detach=True)
        
        #if encoding == 'base64':
        r = r.data

        hoba.signature = r
        return hoba


    def process_auth_response(self, header, encoding='base64'):
        pass


    def get_auth_request(self):
        if self.session_token != None:
            return 'bearer ' + self.session_token

