# standard imports
import logging
import base64
import hashlib
import re

# local imports
from ..auth import Auth


class HTTPAuthorization(Auth):
    """Handles HTTP authorization methods.

    Currently implemented schemes are Basic, Bearer and HOBA. The scheme is determined by the auth string. 

    The digest value used to retrieve authentication data depends on the scheme:

    - Basic: sha256(user:pass), or sha256(user:pass, client_id) if client_id is supplied
    - HOBA: The public key of the client
    - Bearer: The token value

    :param fetcher: Callback used to retrieve authentication data based on the authorization string digest
    :type fetcher: function
    :param http_auth_string: The HTTP "Authorization" header value
    :type http_auth_string: str
    :param realm: Authentication realm
    :type realm: str
    :param client_id: Oauth client_id of entity performing authentication, optional
    :type client_id: str
    """
    re = r'^(\w+) (.+)$'
    logger = logging.getLogger(__name__)
    component_id = None

    def __init__(self, fetcher, http_auth_string, realm, client_id=None):
        self.realm = realm
        m = re.match(self.re, http_auth_string)
        auth_string = None
        if m == None:
            self.logger.debug('invalid authorization string')
        elif m[1].lower() == 'basic':
            (auth_string, retriever) = self.__handle_basic(m[2], fetcher, client_id)
        elif m[1].lower() == 'hoba':
            (auth_string, retriever) = self.__handle_hoba(m[2], fetcher, client_id)
        elif m[1].lower() == 'bearer':
            (auth_string, retriever) = self.__handle_bearer(m[2], fetcher, client_id)
        else:
            self.logger.warning('unhandled authorization scheme {}'.format(m[0]))
       
        super(HTTPAuthorization, self).__init__(fetcher, auth_string, realm)


    # Handler code for "Authorization: Bearer" scheme
    def __handle_bearer(self, s, f, c):
        HTTPAuthorization.component_id = 'http-bearer'
        self.logger.debug('set component id {}'.format(self.component_id))
        return (bytes.fromhex(s), f)


    # Handler code for "Authorization: HOBA" scheme
    def __handle_hoba(self, s, f, c):
        raise NotImplementedError


    # Handler code for "Authorization: Basic" scheme
    def __handle_basic(self, s, f, c):
        HTTPAuthorization.component_id = 'http-basic'
        self.logger.debug('set component id {}'.format(self.component_id))
        auth_string_bytes = base64.b64decode(s)
        h = hashlib.sha256()
        h.update(auth_string_bytes)
        if c != None:
            h.update(c.encode('utf-8'))
        auth_string = h.digest()
        (u, p) = auth_string_bytes.decode('utf-8').split(':')
        return (auth_string, f) 


    """Implements Auth.check
    """
    def check(self):
        self.logger.debug('{} check'.format(HTTPAuthorization.component_id))
        if self.auth_string == None:
            self.logger.debug('no auth string for {}'.format(self.component_id))
            return None
        auth_data = self.retriever.load(self.auth_string)
        if auth_data == None:
            return None
        return (self.component_id, self.auth_string, auth_data)


    """Implements Auth.method
    """
    def method(self):
        s = 'Basic charset="UTF-8"'
        if self.realm != None:
            s += ', realm="{}"'.format(self.realm)
        return s
