# standard import
import logging

# local imports
from cic_auth.error import AuthenticationError

logg = logging.getLogger(__name__)


class AuthAdapter:
    """Holds a collection of authentication methods to be attempted. All members of the collection are expected to implement the cic_auth.Auth interface.

    All activated authenticators will be tried in order that they are added, and will return on the first method that succeeds.
    """
    def __init__(self):
        self.authenticators = {}
        self.authenticators_active = []


    """Implements AuthAdapter.register
    """
    def register(self, authenticator):
        self.authenticators[authenticator.component_id] = authenticator


    """Implements AuthAdapter.activate
    """
    def activate(self, component_id):
        self.authenticators_active.append(component_id)


    """Implements AuthAdapter.check
    """
    def check(self):
        t = None
        for k in self.authenticators_active:
            authenticator = self.authenticators[k]
            try:
                t = authenticator.check()
                if t:
                    logg.info('authentication success {} {}'.format(k, t))
                    return t
            except AuthenticationError as e:
                logg.debug('authentication failed: {}'.format(authenticator))
        return None


    """Implements AuthAdapter.method
    """
    def method(self):
        m = []
        for k in self.authenticators_active:
            m.append(self.authenticators[k].method())
        return m
