# standard import
import logging

# local imports
from cic_auth.error import AuthenticationError

logg = logging.getLogger(__name__)


class Auth:

    component_id = None

    """Base class and interface for authentication backends

    :param retriever: Retriever object to access authentication data
    :type retriever: function
    :param auth_string: Authentication string
    :type auth_string: str
    :param realm: Authentication realm, optional
    :type realm: str
    """
    def __init__(self, retriever, auth_string, realm=None):
        self.retriever = retriever
        self.auth_string = auth_string
        self.realm = realm

    """Interface method.

    In implementation it should execute the authentication lookup with the current state of data.

    :returns: None on failure, or tuple (authentication component id, authentication string, authentication data)
    :rtype: tuple or None
    """
    def check(self):
        return None


class AuthVector:
    """Holds a collection of authentication methods to be attempted. All members of the collection are expected to implement the cic_auth.Auth interface.

    All activated authenticators will be tried in order that they are added, and will return on the first method that succeeds.
    """
    def __init__(self):
        self.authenticators = {}
        self.authenticators_active = []


    """Implements AuthAdapter.register
    """
    def register(self, authenticator):
        self.authenticators[authenticator.component_id] = authenticator


    """Implements AuthAdapter.activate
    """
    def activate(self, component_id):
        self.authenticators_active.append(component_id)


    """Implements AuthAdapter.check
    """
    def check(self):
        t = None
        for k in self.authenticators_active:
            authenticator = self.authenticators[k]
            try:
                t = authenticator.check()
                if t:
                    logg.info('authentication success {} {}'.format(k, t))
                    return t
            except AuthenticationError as e:
                logg.debug('authentication failed: {}'.format(authenticator))
        return None
