# standard imports
import datetime
import logging

# third-party imports
from sqlalchemy import Column, Enum, String, Integer, DateTime, Text, or_
from sqlalchemy.ext.hybrid import hybrid_property, hybrid_method

# local imports
from .base import SessionBase
from cic_eth.db.enum import StatusEnum
from cic_eth.db.error import TxStateChangeError
#from cic_eth.eth.util import address_hex_from_signed_tx

logg = logging.getLogger()


class Otx(SessionBase):
    __tablename__ = 'otx'

    nonce = Column(Integer)
    date_created = Column(DateTime, default=datetime.datetime.utcnow)
    tx_hash = Column(String(66))
    signed_tx = Column(Text)
    status = Column(Integer)
    block = Column(Integer)


    def set_block(self, block):
        if self.block != None:
            raise TxStateChangeError('Attempted set block {} when block was already {}'.format(block, self.block))
        self.block = block


    def waitforgas(self):
        if self.status >= StatusEnum.SENT.value:
            raise TxStateChangeError('WAITFORGAS cannot succeed final state, had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.WAITFORGAS


    def fubar(self):
        self.status = StatusEnum.FUBAR


    def retry(self):
        if self.status != StatusEnum.SENT.value and self.status != StatusEnum.SENDFAIL.value:
            raise TxStateChangeError('RETRY must follow SENT or SENDFAIL, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.RETRY


    def sent(self):
        if self.status > StatusEnum.SENT:
            raise TxStateChangeError('SENT after {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.SENT


    def sendfail(self):
        if self.status not in [StatusEnum.PENDING, StatusEnum.SENT]:
            raise TxStateChangeError('SENDFAIL must follow SENT or PENDING, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.SENDFAIL


    def minefail(self, block=None):
        if block != None:
            self.block = block
        #if self.status != StatusEnum.PENDING and self.status != StatusEnum.OBSOLETED and self.status != StatusEnum.SENT:
        if self.status > StatusEnum.SENT:
            raise TxStateChangeError('REVERTED must follow OBSOLETED, PENDING or SENT, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.REVERTED


    def cancel(self, confirmed=False):
        status = None
        if confirmed:
            if self.status != StatusEnum.OBSOLETED:
                raise TxStateChangeError('CANCELLED must follow OBSOLETED, but had {}'.format(StatusEnum(self.status).name))
            self.status = StatusEnum.CANCELLED
        elif self.status != StatusEnum.OBSOLETED:
            if self.status > StatusEnum.SENT:
                raise TxStateChangeError('OBSOLETED must follow PENDING, SENDFAIL or SENT, but had {}'.format(StatusEnum(self.status).name))
            self.status = StatusEnum.OBSOLETED


    def success(self, block):
        if block != None:
            self.block = block
        if self.status != StatusEnum.SENT:
            raise TxStateChangeError('SUCCESS must follow SENT, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.SUCCESS
        logg.debug('success {}'.format(self.status))


    @staticmethod
    def get(status=0, limit=4096, status_exact=True):
        e = None
        session = Otx.create_session()
        if status_exact:
            e = session.query(Otx.tx_hash).filter(Otx.status==status).order_by(Otx.date_created.asc()).limit(limit).all()
        else:
            e = session.query(Otx.tx_hash).filter(Otx.status<=status).order_by(Otx.date_created.asc()).limit(limit).all()
        session.close()
        return e


    @staticmethod
    def load(tx_hash):
        session = Otx.create_session()
        q = session.query(Otx)
        q = q.filter(Otx.tx_hash==tx_hash)
        session.close()
        return q.first()


    @staticmethod
    def account(account_address):
        session = Otx.create_session()
        q = session.query(Otx.tx_hash)
        q = q.join(TxCache)
        q = q.filter(or_(TxCache.sender==account_address, TxCache.recipient==account_address))
        txs = q.all()
        session.close()
        return list(txs)


    def __init__(self, nonce, address, tx_hash, signed_tx):
        self.nonce = nonce
        self.tx_hash = tx_hash
        self.signed_tx = signed_tx
        self.status = StatusEnum.PENDING
        signed_tx_bytes = bytes.fromhex(signed_tx[2:])
       # sender_address = address_hex_from_signed_tx(signed_tx_bytes)
       # logg.debug('decoded tx {}'.format(sender_address))


class OtxSync(SessionBase):
    __tablename__ = 'otx_sync'

    blockchain = Column(String)
    block_height_backlog = Column(Integer)
    tx_height_backlog = Column(Integer)
    block_height_session = Column(Integer)
    tx_height_session = Column(Integer)
    block_height_head = Column(Integer)
    tx_height_head = Column(Integer)
    date_created = Column(DateTime, default=datetime.datetime.utcnow)
    date_updated = Column(DateTime)


    def backlog(self, block_height=None, tx_height=None):
        #session = OtxSync.create_session()
        if block_height != None:
            if tx_height == None:
                raise ValueError('tx height missing')
            self.block_height_backlog = block_height
            self.tx_height_backlog = tx_height
            #session.add(self)
            self.date_updated = datetime.datetime.utcnow()
        #session.commit()
        block_height = self.block_height_backlog
        tx_height = self.tx_height_backlog
        #session.close()
        return (block_height, tx_height)
   

    def session(self, block_height=None, tx_height=None):
        #session = OtxSync.create_session()
        if block_height != None:
            if tx_height == None:
                raise ValueError('tx height missing')
            self.block_height_session = block_height
            self.tx_height_session = tx_height
            #session.add(self)
            self.date_updated = datetime.datetime.utcnow()
        #session.commit()
        block_height = self.block_height_session
        tx_height = self.tx_height_session
        #session.close()
        return (block_height, tx_height)


    def head(self, block_height=None, tx_height=None):
        #session = OtxSync.create_session()
        if block_height != None:
            if tx_height == None:
                raise ValueError('tx height missing')
            self.block_height_head = block_height
            self.tx_height_head = tx_height
            #session.add(self)
            self.date_updated = datetime.datetime.utcnow()
        #session.commit()
        block_height = self.block_height_head
        tx_height = self.tx_height_head
        #session.close()
        return (block_height, tx_height)


    @hybrid_property
    def synced(self):
        #return self.block_height_session == self.block_height_backlog and self.tx_height_session == self.block_height_backlog
        return self.block_height_session == self.block_height_backlog and self.tx_height_session == self.tx_height_backlog


    @staticmethod
    def load(blockchain_string, session):
        q = session.query(OtxSync)
        q = q.filter(OtxSync.blockchain==blockchain_string)
        return q.first()


    @staticmethod
    def latest(nonce):
        session = SessionBase.create_session()
        otx = session.query(Otx).filter(Otx.nonce==nonce).order_by(Otx.created.desc()).first()
        session.close()
        return otx


    @staticmethod
    def get_expired(datetime_threshold):
        session = SessionBase.create_session()
        q = session.query(Otx)
        q = q.filter(Otx.date_created<datetime_threshold)
        q = q.filter(Otx.status==StatusEnum.SENT)
        q = q.order_by(Otx.date_created.desc())
        q = q.group_by(Otx.nonce)
        q = q.group_by(Otx.id)
        otxs = q.all()
        session.close()
        return otxs


    def chain(self):
        return self.blockchain


    def __init__(self, blockchain):
        self.blockchain = blockchain
        self.block_height_head = 0
        self.tx_height_head = 0
        self.block_height_session = 0
        self.tx_height_session = 0
        self.block_height_backlog = 0
        self.tx_height_backlog = 0
