# standard imports
import logging

# local imports
from cic_eth.db.models.sync import BlockchainSync
from cic_eth.db.models.base import SessionBase

logg = logging.getLogger()


class SyncerBackend:

    def __init__(self, chain_spec, object_id):
        self.db_session = None
        self.db_object = None
        self.chain_spec = chain_spec
        self.object_id = object_id
        self.connect()
        self.disconnect()


    def connect(self):
        self.db_session = SessionBase.create_session()
        q = self.db_session.query(BlockchainSync)
        q = q.filter(BlockchainSync.id==self.object_id)
        self.db_object = q.first()
        if self.db_object == None:
            raise ValueError('sync entry with id {} not found'.format(self.object_id))


    def disconnect(self):
        self.db_session.add(self.db_object)
        self.db_session.commit()
        self.db_session.close()
       

    def chain(self):
        return self.chain_spec
   

    def get(self):
        self.connect()
        pair = self.db_object.cursor()
        self.disconnect()
        return pair
   

    def set(self, block_height, tx_height):
        self.connect()
        pair = self.db_object.set(block_height, tx_height)
        self.disconnect()
        return pair


    def start(self):
        self.connect()
        pair = self.db_object.start()
        self.disconnect()
        return pair

    
    def target(self):
        self.connect()
        target = self.db_object.target()
        self.disconnect()
        return target


    @staticmethod
    def first(chain):
        return BlockchainSync.first(chain)


    @staticmethod
    def initial(chain, block_height):
        object_id = None
        session = SessionBase.create_session()
        o = BlockchainSync(chain, 0, 0, block_height)
        session.add(o)
        session.commit()
        object_id = o.id
        session.close()

        return SyncerBackend(chain, object_id)


    @staticmethod
    def resume(chain, block_height):
        syncers = []

        session = SessionBase.create_session()

        object_id = None

        for object_id in BlockchainSync.get_unsynced(session=session):
            logg.debug('block syncer resume added previously unsynced sync entry id {}'.format(object_id))
            syncers.append(SyncerBackend(chain, object_id))

        (block_resume, tx_resume) = BlockchainSync.get_last_live_height(block_height, session=session)
        if block_height != block_resume:
            o = BlockchainSync(chain, block_resume, tx_resume, block_height)
            session.add(o)
            session.commit()
            object_id = o.id
            syncers.append(SyncerBackend(chain, object_id))
            logg.debug('block syncer resume added new sync entry from previous run id {}, start{}:{} target {}'.format(object_id, block_resume, tx_resume, block_height))

        session.close()

        return syncers


    @staticmethod
    def live(chain, block_height):
        object_id = None
        session = SessionBase.create_session()
        o = BlockchainSync(chain, block_height, 0, None)
        session.add(o)
        session.commit()
        object_id = o.id
        session.close()

        return SyncerBackend(chain, object_id)
