"""
.. module:: tx
   :synopsis: Transaction queue tasks

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# standard imports
import logging
import time

# third-party imports
import celery

# local imports
from cic_eth.db.models.otx import Otx
from cic_eth.db.models.tx import Tx
from cic_eth.db.models.tx import TxCache
from cic_eth.db import SessionBase
from cic_eth.db.enum import StatusEnum
from cic_eth.error import NotLocalTxError

#celery_app = celery.Celery(broker='redis://')
celery_app = celery.current_app
#logg = celery_app.log.get_default_logger()
logg = logging.getLogger()


@celery_app.task()
def create(nonce, holder_address, tx_hash, signed_tx):
    """Create a new transaction queue record.

    :param nonce: Transaction nonce
    :type nonce: int
    :param holder_address: Sender address
    :type holder_address: str, 0x-hex
    :param tx_hash: Transaction hash
    :type tx_hash: str, 0x-hex
    :param signed_tx: Signed raw transaction
    :type signed_tx: 0x-hex
    :returns: transaction hash
    :rtype: str, 0x-hash
    """
    session = SessionBase.create_session()
    o = Otx(
            nonce=nonce,
            address=holder_address,
            tx_hash=tx_hash,
            signed_tx=signed_tx,
            )
    session.add(o)
    for otx in session.query(Otx).filter(Otx.nonce==nonce).all():
        if otx.tx_hash != tx_hash:
            otx.cancel(False)
            session.add(otx)
    session.commit()
    session.close()
    return tx_hash


@celery_app.task()
def set_sent_status(tx_hash, fail=False):
    """Used to set the status after a send attempt

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :param fail: if True, will set a SENDFAIL status, otherwise a SENT status. (Default: False)
    :type fail: boolean
    :raises NotLocalTxError: If transaction not found in queue.
    :returns: True if tx is known, False otherwise
    :rtype: boolean
    """
    session = SessionBase.create_session()
    o = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if o == None:
        logg.warning('not local tx, skipping {}'.format(tx_hash))
        return False

    if fail:
        o.sendfail()
    else:
        o.sent()

    session.add(o)
    session.commit()
    session.close()
    return True


@celery_app.task()
def set_final_status(tx_hash, block=None, fail=False):
    """Used to set the status of an incoming transaction result. 

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :param fail: if True, will set a SUCCESS status, otherwise a REVERTED status. (Default: False)
    :type fail: boolean
    :raises NotLocalTxError: If transaction not found in queue.
    """
    session = SessionBase.create_session()
    o = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if o == None:
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    if fail:
        o.minefail(block)
    else:
        o.success(block)

    session.add(o)
    session.flush()

    q = session.query(Otx).filter(Otx.nonce==o.nonce)
    for otwo in session.query(Otx).filter(Otx.nonce==o.nonce).all():
        if otwo.id != o.id:
            otwo.cancel(True)
            session.add(otwo)

    session.commit()
    session.close()


@celery_app.task()
def set_fubar(tx_hash):
    """Used to set the status when an unexpected error occurs.

    Will set the state to FUBAR

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :raises NotLocalTxError: If transaction not found in queue.
    """

    session = SessionBase.create_session()
    o = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if o == None:
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    o.fubar()
    session.add(o)
    session.commit()
    session.close()


def get_tx(tx_hash):
    """Retrieve a transaction queue record by transaction hash

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :raises NotLocalTxError: If transaction not found in queue.
    :returns: nonce, address and signed_tx (raw signed transaction)
    :rtype: dict
    """
    session = SessionBase.create_session()
    tx = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if tx == None:
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    txc = session.query(TxCache).filter(TxCache.otx_id==tx.id).first()
    if txc == None:
        raise AttributeError('otx {} missing tx cache link, tx hash {}'.format(tx.id, tx_hash))

    o = {
        'otx_id': tx.id,
        'cache_id': txc.id,
        'nonce': tx.nonce,
        'address': txc.sender,
        'signed_tx': tx.signed_tx,
            }
    logg.debug('get tx {}'.format(o))
    session.close()
    return o


@celery_app.task()
def incoming(tx_hash, processing_task, success=True, tx_data=None):
    session = SessionBase.create_session()
    logg.debug('tx_data {}'.format(tx_data))
    tx = Tx(tx_hash, success)
    session.add(tx)
    session.commit()

    tx_cache_id = None
    try:
        tx_data = get_tx(tx_hash)
        tx_cache_id = tx_data['cache_id']
        logg.debug('tx found: {} {} {} {}'.format(tx.id, tx.tx_hash, tx_data, processing_task))
    except NotLocalTxError as e:
        logg.debug('incoming tx not found: {} {} {} {}'.format(tx, e, tx_data, processing_task))
        s = celery.signature(
                processing_task,
                [tx_hash, tx_data, False],
                )
        t = s.apply()
        r = t.get()
        tx_cache_id = r[1]
    
    txc = session.query(TxCache).get(tx_cache_id)
    txc.tx_id = tx_cache_id
    session.add(txc)
    session.commit()
    session.close()

    return tx_hash
